﻿using System;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class RemoteStorageMoqTests
    {
        private class Content
        {
            public int Id { get; set; }
        }

        private class RemoteStorageMoq : RemoteStorage<Content>
        {
            public TimeSpan DelayForDownload { get; set; } = TimeSpan.FromSeconds(0.1d);
            public StorageResultCode ResultCodeForNextDownload { get; set; } = StorageResultCode.Success;
            public Content ResultForNextDownload { get; set; } = new();

            public RemoteStorageMoq(string storagePath = "moq_remote_storage") : base(storagePath, EditorModeTestsEnvironment.RootStoragePath)
            {
            }

            protected override Content LoadFromDisk(string url)
            {
                return new Content();
            }

            protected override async UniTask<(StorageResultCode Code, Content Value)> DownloadFromUrlWithoutTimeout([NotNull] string url, CancellationToken cancellationToken)
            {
                await UniTask.Delay(DelayForDownload, cancellationToken: cancellationToken);
                return (ResultCodeForNextDownload, ResultForNextDownload);
            }

            protected override void ReleaseContent(Content cache)
            {
            }

            protected override bool HasInternetConnection()
            {
                return true;
            }
        }

        [Test]
        public async Task CreateMoq_NoDelay_Load_HasCorrectResult()
        {
            // Arrange
            var remoteStorage = new RemoteStorageMoq();
            var expectedResultCode = remoteStorage.ResultCodeForNextDownload = StorageResultCode.Success;
            var expectedResult = remoteStorage.ResultForNextDownload = new Content() { Id = 413 };

            // Act
            var result = await remoteStorage.Load(string.Empty, 1d, CancellationToken.None);

            // Assert
            result.Code.Should().Be(expectedResultCode);
            result.Value.Id.Should().Be(expectedResult.Id);
        }

        [Test]
        [TestCase(0.01d, true)]
        [TestCase(0.03d, true)]
        [TestCase(0.3d, false)]
        [TestCase(1.0d, false)]
        public async Task CreateMoq_Delay_Load_HasCorrectResult(double loadingDelay, bool mustHaveResult)
        {
            // Arrange
            var remoteStorage = new RemoteStorageMoq() { DelayForDownload = TimeSpan.FromSeconds(loadingDelay) };
            var expectedResultCode = remoteStorage.ResultCodeForNextDownload = StorageResultCode.Success;
            var expectedResult = remoteStorage.ResultForNextDownload = new Content() { Id = 413 };

            // Act
            var result = await remoteStorage.Load(string.Empty, 0.1d, CancellationToken.None);

            // Assert
            if (mustHaveResult)
            {
                result.Code.Should().Be(expectedResultCode);
                result.Value.Id.Should().Be(expectedResult.Id);
            }
            else
            {
                result.Code.Should().Be(StorageResultCode.Timeout);
                result.Value.Should().Be(null);
            }
        }

        [Test]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(100)]
        public async Task CreateMoq_Delay_ManyLoadings_HaveCorrectResult(int loadingsNumber)
        {
            // Arrange
            var remoteStorage = new RemoteStorageMoq() { DelayForDownload = TimeSpan.FromSeconds(0.1d) };
            var expectedResultCode = remoteStorage.ResultCodeForNextDownload = StorageResultCode.Success;
            var expectedResult = remoteStorage.ResultForNextDownload = new Content() { Id = 413 };
            var results = new ContentHandle<Content>[loadingsNumber];

            // Act
            for (var i = 0; i < loadingsNumber; i++)
            {
                var index = i;
                remoteStorage.Load(string.Empty, 1d, CancellationToken.None).ContinueWith(r => results[index] = r).Forget();
            }

            await UniTask.Delay(TimeSpan.FromSeconds(0.3d));

            // Assert
            results.ForEach(r => r.Code.Should().Be(expectedResultCode));
            results.ForEach(r => r.Value.Id.Should().Be(expectedResult.Id));
        }

        [Test]
        [TestCase(2, 1)]
        [TestCase(5, 3)]
        [TestCase(100, 69)]
        public async Task CreateMoq_Delay_ManyLoadings_DifferentTimeout_HaveCorrectResult(int loadingsNumber, int withLessTimeout)
        {
            // Arrange
            var remoteStorage = new RemoteStorageMoq() { DelayForDownload = TimeSpan.FromSeconds(0.1d) };
            var expectedResultCode = remoteStorage.ResultCodeForNextDownload = StorageResultCode.Success;
            var expectedResult = remoteStorage.ResultForNextDownload = new Content() { Id = 413 };
            var results = new ContentHandle<Content>[loadingsNumber];

            // Act
            for (var i = 0; i < loadingsNumber; i++)
            {
                var index = i;
                var timeout = index < withLessTimeout ? 0.01d : 1d;
                remoteStorage.Load(string.Empty, timeout, CancellationToken.None).ContinueWith(r => results[index] = r).Forget();
            }

            await TaskScheduler.SwitchToMainThread(CancellationToken.None);
            await UniTask.Delay(TimeSpan.FromSeconds(0.3d));

            // Assert
            results.Count(r => r.Value == null).Should().Be(withLessTimeout);
            results.Count(r => r.Code == StorageResultCode.Timeout).Should().Be(withLessTimeout);

            var finished = results.Where(r => r.Value != null).ToArray();
            finished.Length.Should().Be(loadingsNumber - withLessTimeout);
            finished.ForEach(r => r.Code.Should().Be(expectedResultCode));
            finished.ForEach(r => r.Value.Id.Should().Be(expectedResult.Id));
        }

        [Test]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(100)]
        public async Task CreateMoq_ManyMultithreadingLoadings_HaveCorrectResult(int loadingsNumber)
        {
            // Arrange
            var remoteStorage = new RemoteStorageMoq() { DelayForDownload = TimeSpan.FromSeconds(0.1d) };
            var expectedResultCode = remoteStorage.ResultCodeForNextDownload = StorageResultCode.Success;
            var expectedResult = remoteStorage.ResultForNextDownload = new Content() { Id = 413 };
            var results = new ContentHandle<Content>[loadingsNumber];
            var cancellationToken = new CancellationTokenSource(1000);

            // Act
            try
            {
                for (var i = 0; i < loadingsNumber; i++)
                {
                    var index = i;
                    UniTask.RunOnThreadPool(() =>
                    {
                        remoteStorage.Load(string.Empty, 1d, cancellationToken.Token)
                            .ContinueWith(r => results[index] = r)
                            .Forget();
                    }, cancellationToken: cancellationToken.Token).Forget();
                }
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
                return;
            }

            await UniTask.Delay(TimeSpan.FromSeconds(0.3d), cancellationToken: cancellationToken.Token);

            // Assert
            results.ForEach(r => r.Code.Should().Be(expectedResultCode));
            results.ForEach(r => r.Value.Id.Should().Be(expectedResult.Id));
        }
    }
}