﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Magify.Features;
using UnityEditor;

namespace Magify.Tests
{
    public static class TestCoverage
    {
        private static readonly Type[] _checkTargets =
        {
            typeof(SessionCounter),
            typeof(GeneralPrefs),
            typeof(ServerApi),
            typeof(AnalyticsTracker),
            typeof(AnalyticsSessionTracker),
            // typeof(AnalyticsService), => just a wrapper over CampaignsTracker and AnalyticsTracker
            typeof(CampaignsTracker),
            typeof(CampaignsProvider),
            typeof(LimitsHolder),
            typeof(FeaturesProvider),
            typeof(DefaultProductsContainer),
            typeof(ContentProvider),
            typeof(CampaignUpdateHandler)
        };

        [MenuItem("Tools/TestCoverageCheck")]
        public static (List<string> Tested, List<string> NotTested) Check()
        {
            var assemblies = AppDomain.CurrentDomain.GetAssemblies();
            var testsTypes = assemblies.First(a => a.GetName().Name == "Magify.CSharpTests").GetTypes();
            var existingTests = new HashSet<string>();
            foreach (var method in testsTypes)
            {
                var methods = method.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic);
                foreach (var methodInfo in methods)
                {
                    var targetAttributes = methodInfo.GetCustomAttributes<UnitTestTargetAttribute>();
                    foreach (var targetAttribute in targetAttributes)
                    {
                        if (targetAttribute is { Type: not null } && !string.IsNullOrWhiteSpace(targetAttribute.MethodName))
                        {
                            existingTests.Add(targetAttribute.GetMethodFullPath());
                        }
                    }
                }
            }

            var targets = new List<(string Type, MethodInfo Method)>();
            foreach (var type in _checkTargets)
            {
                if (IsPublic(type) || IsInternal(type))
                {
                    var publics = type.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly);
                    var internals = type.GetMethods(BindingFlags.Instance | BindingFlags.NonPublic | BindingFlags.DeclaredOnly).Where(m => m.IsAssembly);
                    targets.AddRange(publics
                        .Concat(internals)
                        .Where(m => !m.IsSpecialName && !m.Name.Contains("Reset"))
                        .Select(methodInfo => (type.FullName, methodInfo)));
                }
            }

            var testedMethods = new List<string>();
            var notTestedMethods = new List<string>();
            foreach (var method in targets.Select(t => $"{t.Type}.{t.Method.Name}"))
            {
                if (existingTests.Contains(method))
                {
                    testedMethods.Add(method);
                }
                else
                {
                    notTestedMethods.Add(method);
                }
            }

            return (testedMethods, notTestedMethods);
        }

        private static bool IsPublic(Type t)
        {
            return
                t.IsVisible
             && t.IsPublic
             && !t.IsNotPublic
             && !t.IsNested
             && !t.IsNestedPublic
             && !t.IsNestedFamily
             && !t.IsNestedPrivate
             && !t.IsNestedAssembly
             && !t.IsNestedFamORAssem
             && !t.IsNestedFamANDAssem;
        }

        public static bool IsInternal(Type t)
        {
            return
                !t.IsVisible
             && !t.IsPublic
             && t.IsNotPublic
             && !t.IsNested
             && !t.IsNestedPublic
             && !t.IsNestedFamily
             && !t.IsNestedPrivate
             && !t.IsNestedAssembly
             && !t.IsNestedFamORAssem
             && !t.IsNestedFamANDAssem;
        }
    }
}