using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class SaveProgress : ServerApiTestsPart
        {
            private static object[] DefaultTestCases =>
                new object[]
                {
                    new object[] { RuntimePlatform.Android },
                    new object[] { RuntimePlatform.IPhonePlayer },
                };

            private static object[] SaveProgressResultCases =>
                new object[]
                {
                    new object[] { ErrorCode.Expired, RuntimePlatform.Android },
                    new object[] { ErrorCode.Expired, RuntimePlatform.IPhonePlayer },
                    new object[] { ErrorCode.Banned, RuntimePlatform.Android },
                    new object[] { ErrorCode.Banned, RuntimePlatform.IPhonePlayer },
                    new object[] { ErrorCode.Revoked, RuntimePlatform.Android },
                    new object[] { ErrorCode.Revoked, RuntimePlatform.IPhonePlayer },
                };

            private static object[] MultiplyLoadingsCases =>
                new object[]
                {
                    new object[] { 2, RuntimePlatform.Android },
                    new object[] { 2, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 100, RuntimePlatform.Android },
                    new object[] { 100, RuntimePlatform.IPhonePlayer },
                };

            private static object[] CancellationTokenCancelledCases =>
                new object[]
                {
                    new object[] { true, true, RuntimePlatform.Android },
                    new object[] { true, true, RuntimePlatform.IPhonePlayer },
                    new object[] { false, false, RuntimePlatform.Android },
                    new object[] { false, false, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RequestWithDelayCases =>
                new object[]
                {
                    new object[] { 10, RuntimePlatform.Android },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.Android },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                    new object[] { 200, RuntimePlatform.IPhonePlayer },
                };

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenSaveProgress_AndCancelAllSaveStateRequests_ThenGetOperationCancelledException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var moqReached = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    moqReached = true;
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.WaitUntil(() => moqReached, cancellationToken: cancellationToken);
                        serverApi.CancelAllSaveStateRequests();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SaveState(string.Empty, null, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(MultiplyLoadingsCases))]
            public async Task WhenMultiplyLoadingSaveProgress_ThenRequestToNetworkShouldBeSameLoadingsNumber(int loadingsNumber, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (message.Method == ApplicationStateApiMethods.SaveProgress(runtimePlatform))
                    {
                        requestCount++;
                        promise.TrySetResult();
                    }
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    var tasks = loadingsNumber.EnumerateEach().Select(_ => serverApi.SaveState(string.Empty, null, cancellationToken).AsUniTask());
                    await UniTask.WhenAll(tasks);
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenServerApiDisposed_ThenSaveProgressIsCancelled(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.SaveState(string.Empty, null, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenSaveProgress_WithCancelledToken_ThenGetException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var cancellationToken = GetCancellationToken(50);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await cancellationToken.WaitUntilCanceled();
                    await serverApi.SaveState(string.Empty, null, cancellationToken);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenSaveProgress_AndEndpointUrlIsOffline_ThenSaveProgressResultShouldBeFail(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline, platformAPI : runtimePlatform);
                systems.InitializeAll();

                // Act
                var saveProgressResult = await serverApi.SaveState(string.Empty, null, cancellationToken);

                // Assert
                saveProgressResult.Status.Should()!.Be(SaveStateRequestResponse.StatusCode.Fail);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(CancellationTokenCancelledCases))]
            public async Task WhenSaveProgress_AndCancellationTokenWasCancelled_ThenGetException(bool mustBeCanceled, bool expectedError, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                cts.CancelAfter(mustBeCanceled ? 1 : 1000);
                try
                {
                    await serverApi.SaveState(string.Empty, null, cts.Token);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
                cts.IsCancellationRequested.Should()!.Be(mustBeCanceled);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(RequestWithDelayCases))]
            public async Task WhenSaveProgressWithDelay_ThenResultShouldBeTrue(int delayMs, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var saveProgressResult = await serverApi.SaveState(string.Empty, null, cancellationToken);

                // Assert
                saveProgressResult.Status.Should()!.Be(SaveStateRequestResponse.StatusCode.Success);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(SaveProgressResultCases))]
            public async Task WhenSaveProgress_AndGetResponseErrorResult_ThenPurchaseVerificationResultCodeShouldBeSame(ErrorCode errorCode, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = errorCode } });
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();
                LogAssert.ignoreFailingMessages = true;

                // Act
                var saveProgressResult = await serverApi.SaveState(string.Empty, null, cancellationToken);

                // Assert
                saveProgressResult.Status.Should()!.NotBe(SaveStateRequestResponse.StatusCode.Success);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}