using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MinimalAnalyticsTrackerTests
    {
        public delegate Action<object> CreateEventRequestHandlerDelegate(ReactiveProperty<bool?> isEventBasic);

        [NotNull, ItemNotNull]
        private static readonly List<CancellationTokenSource> _cancellationTokenSources = new();

        private static object[] IsBasicIntegrationCases =>
            new object[]
            {
                new object[] {
                    EventType.AppLaunch,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var appLaunchEvent = @event as AppLaunchEvent;
                            isEventBasic.Value = appLaunchEvent?.IsBasicIntegration is true;
                        });
                    }),
                    (Action<MinimalAnalyticsTracker>)AnalyticsTrackerUtils.TrackAppLaunch },
                new object[] {
                    EventType.AppBackgroundEvent,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var backgroundEvent = @event as AppBackgroundEvent;
                            isEventBasic.Value = backgroundEvent?.IsBasicIntegration is true;
                        });
                    }),
                    (Action<MinimalAnalyticsTracker>)AnalyticsTrackerUtils.TrackBackgroundPurchaseEvent },
                new object[] {
                    EventType.ApplovinAdsImpression,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var adsImpressionEvent = @event as AdsImpressionEvent;
                            isEventBasic.Value = adsImpressionEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<MinimalAnalyticsTracker>)AnalyticsTrackerUtils.TrackAdsImpression },
                new object[] {
                    EventType.InApp,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var productPurchaseEvent = @event as ProductPurchaseEvent;
                            isEventBasic.Value = productPurchaseEvent?.IsBasicIntegration is true;
                        });
                    }),
                    (Action<MinimalAnalyticsTracker>)AnalyticsTrackerUtils.TrackInAppPurchaseEvent },
            };

        [Test]
        [TestCaseSource(nameof(IsBasicIntegrationCases))]
        public async Task WhenTrackEvent_ThenIsBasicPropertyShouldBeTrue(EventType eventType, [NotNull] CreateEventRequestHandlerDelegate createEventRequestHandler, Action<MinimalAnalyticsTracker> trackAction)
        {
            //Arrange
            var isEventBasic = new ReactiveProperty<bool?>(null);
            var handleEventRequest = createEventRequestHandler(isEventBasic);
            using var systems = AnalyticsTrackerUtils.Create(eventType, handleEventRequest, out MinimalAnalyticsTracker analyticsTracker, out var serverApi);
            systems.InitializeAll();
            analyticsTracker.SetupAnalyticsConfig(new AnalyticsConfiguration());
            await serverApi.CancelAllServerInteractions();

            //Act
            var cancellationToken = GetCancellationToken();
            var task = UniTask.WhenAny(
                isEventBasic.WaitUntilValueChangedAsync(cancellationToken),
                UniTask.Delay(1000, cancellationToken: cancellationToken));
            trackAction?.Invoke(analyticsTracker);
            await task;

            //Assert
            isEventBasic.Value.Should()!.NotBeNull();
            isEventBasic.Value.Should()!.BeTrue();
        }

        [SetUp]
        [TearDown]
        public void ClearAllData()
        {
            foreach (var cts in _cancellationTokenSources)
            {
                cts.Cancel();
                cts.Dispose();
            }
            _cancellationTokenSources.Clear();
            EditorModeTestsEnvironment.Clear();
        }

        protected static CancellationToken GetCancellationToken(int cancelDelay = 1000)
        {
            var cancellationTokenSource = new CancellationTokenSource(cancelDelay);
            _cancellationTokenSources.Add(cancellationTokenSource);
            return cancellationTokenSource.Token;
        }
    }
}