﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class LimitsHolderTests
    {
        private static IReadOnlyList<CampaignType> CampaignTypesWithLimits =>
            typeof(Model.Limits)
                .GetMembers(BindingFlags.Default | BindingFlags.Instance | BindingFlags.Public)
                .Where(p => p.GetCustomAttribute<LimitsForCampaignTypeAttribute>() != null)
                .Select(p => p.GetCustomAttribute<LimitsForCampaignTypeAttribute>()!.CampaignType)
                .ToArray();

        [Test]
        [UnitTestTarget(typeof(LimitsHolder), nameof(LimitsHolder.GetCampaignTypeLimits))]
        [TestCase(121, 53, 12)]
        [TestCase(3, 1, 2)]
        public void CallGetCampaignTypeLimits_AndReturnValidLimitsForCampaignType(int global, int session, int daily)
        {
            // Arrange
            using var _ = CreateHolder(out var limitsHolder);
            var context = GenerateContextWithCampaignTypeLimits(global, session, daily);
            ((IContextListener)limitsHolder).UpdateContext(context, ContextKind.Default);
            var impressionLimits = context.Limits!.InAppLimits!;

            // Act
            var limitsByType = limitsHolder.GetCampaignTypeLimits(ArraySegment<TemporaryLimitsGroup>.Empty);
            var campaignTypes = typeof(CampaignType).GetEnumValues()
                .Cast<CampaignType>()
                .Where(c => CampaignTypesWithLimits.Contains(c))
                .Distinct()
                .ToArray();

            // Assert
            limitsByType.Keys
                .Where(c => CampaignTypesWithLimits.Contains(c))
                .ToArray()
                .ShouldBeSameInAnyOrder(campaignTypes);
            limitsByType
                .Where(p => CampaignTypesWithLimits.Contains(p.Key))
                .ForEach(pair =>
                {
                    var because = $"campaign type {pair.Key} supports {{0}} limit";
                    pair.Value.GlobalLimit.Should().Be(impressionLimits.GlobalLimit, because, "global");
                    pair.Value.SessionLimit.Should().Be(impressionLimits.SessionLimit, because, "session");
                    pair.Value.DailyLimit.Should().Be(impressionLimits.DailyLimit, because, "daily");
                });
        }

        [Test]
        [UnitTestTarget(typeof(LimitsHolder), nameof(LimitsHolder.GetCampaignTypeLimits))]
        [TestCase(121, 53, 12)]
        [TestCase(3, 1, 2)]
        public void CallGetCampaignTypeLimits_AndNoReturnLimitsForCampaignTypeWithoutLimits(int global, int session, int daily)
        {
            // Arrange
            using var _ = CreateHolder(out var limitsHolder);
            var context = GenerateContextWithCampaignTypeLimits(global, session, daily);
            ((IContextListener)limitsHolder).UpdateContext(context, ContextKind.Default);
            var impressionLimits = context.Limits!.InAppLimits!;

            // Act
            var limitsByType = limitsHolder.GetCampaignTypeLimits(ArraySegment<TemporaryLimitsGroup>.Empty);
            var campaignTypesWithoutLimits = typeof(CampaignType).GetEnumValues()
                .Cast<CampaignType>()
                .Where(c => !CampaignTypesWithLimits.Contains(c))
                .Distinct()
                .ToArray();

            // Assert
            limitsByType.Keys
                .Where(c => !CampaignTypesWithLimits.Contains(c))
                .ToArray()
                .ShouldBeSameInAnyOrder(campaignTypesWithoutLimits);
            var t =
                limitsByType
                    .Where(p => !CampaignTypesWithLimits.Contains(p.Key));
            limitsByType
                .Where(p => !CampaignTypesWithLimits.Contains(p.Key))
                .ForEach(pair =>
                {
                    var because = $"campaign type {pair.Key} doesn't support {{0}} limits";
                    pair.Value.GlobalLimit.Should().NotHaveValue(because, "global");
                    pair.Value.SessionLimit.Should().NotHaveValue(because, "session");
                    pair.Value.DailyLimit.Should().NotHaveValue(because, "daily");
                });
        }

        [Test]
        [UnitTestTarget(typeof(LimitsExtensions), nameof(LimitsExtensions.GetCampaignTypeImpressionLimits))]
        public void GetCampaignTypeImpressionLimits_ReturnsCorrectLimitsByCampaignType()
        {
            // Arrange
            var limits = new Model.Limits();
            var typeToLimit = new Dictionary<CampaignType, ImpressionLimits>();
            var random = new Random();
            foreach (var campaignType in CampaignTypesWithLimits)
            {
                var campaignLimits = typeToLimit[campaignType] = new ImpressionLimits
                {
                    GlobalLimit = random.Next(),
                    SessionLimit = random.Next(),
                    DailyLimit = random.Next(),
                };
                var member = typeof(Model.Limits)
                    .GetMembers(BindingFlags.Default | BindingFlags.Instance | BindingFlags.Public)
                    .Where(m => m.MemberType is MemberTypes.Field or MemberTypes.Property)
                    .First(m => m.GetCustomAttribute<LimitsForCampaignTypeAttribute>() is { } attribute && attribute.CampaignType == campaignType);
                if (member is PropertyInfo property) property.SetValue(limits, campaignLimits);
                if (member is FieldInfo field) field.SetValue(limits, campaignLimits);
            }

            // Act
            var calculatedTypeToLimit = CampaignTypesWithLimits
                .ToDictionary(
                    c => c,
                    c => limits.GetCampaignTypeImpressionLimits(c));

            // Assert
            calculatedTypeToLimit.Should().HaveSameCount(typeToLimit);
            typeToLimit.ForEach(p =>
            {
                calculatedTypeToLimit.Should().ContainKey(p.Key);
                var calculatedLimit = calculatedTypeToLimit[p.Key];
                var targetLimit = typeToLimit[p.Key];
                calculatedLimit.GlobalLimit.Should().Be(targetLimit.GlobalLimit);
                calculatedLimit.SessionLimit.Should().Be(targetLimit.SessionLimit);
                calculatedLimit.DailyLimit.Should().Be(targetLimit.DailyLimit);
            });

        }

        private static CampaignsContext GenerateContextWithCampaignTypeLimits(int global, int session, int daily)
        {
            var impressionLimits = new ImpressionLimits
            {
                GlobalLimit = global,
                SessionLimit = session,
                DailyLimit = daily,
            };

            var properties = typeof(Model.Limits)
                .GetProperties(BindingFlags.Default | BindingFlags.Instance | BindingFlags.Public)
                .Where(p => p.PropertyType == typeof(ImpressionLimits))
                .ToArray();

            var limits = new Model.Limits();
            properties.ForEach(p => p.SetValue(limits, impressionLimits));
            return new CampaignsContext { Limits = limits };
        }

        private static SubsystemsCollection CreateHolder(out LimitsHolder limitsHolder) => CreateHolder(out limitsHolder, out _, out _);

        private static SubsystemsCollection CreateHolder(out LimitsHolder limitsHolder, out GeneralPrefs generalPrefs, out SessionCounter sessionCounter)
        {
            var subsystems = new SubsystemsCollection();
            var internalConfigPrefs = InternalConfigPrefs.Create(EditorModeTestsEnvironment.InternalConfigPrefsPath).AddTo(subsystems);
            var platform = new PlatformDefault(internalConfigPrefs).AddTo(subsystems);
            generalPrefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider()).AddTo(subsystems);
            sessionCounter = new SessionCounter(generalPrefs, platform).AddTo(subsystems);
            limitsHolder = new LimitsHolder(generalPrefs, sessionCounter).AddTo(subsystems);
            return subsystems;
        }
    }
}