﻿using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Types;
using EventType = Magify.Model.EventType;

namespace Magify.Tests
{
    internal static class AnalyticsTrackerUtils
    {
        internal static void TrackAppLaunch([NotNull] MinimalAnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackAppLaunch();
        }
        internal static void TrackAppLaunch([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackAppLaunch();
        }
        internal static void TrackBackgroundPurchaseEvent([NotNull] MinimalAnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackAppBackgroundEvent(0, 0, "version");
        }
        internal static void TrackBackgroundPurchaseEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackAppBackgroundEvent(0, 0, "version");
        }
        internal static void TrackAdsImpression([NotNull] MinimalAnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackApplovinAdsImpression(GetApplovinAdsImpression());
        }
        internal static void TrackAdsImpression([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackApplovinAdsImpression(GetCampaignImpression(), GetApplovinAdsImpression());
        }
        internal static void TrackInAppPurchaseEvent([NotNull] MinimalAnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackInAppPurchaseEvent(EventType.InApp, GetPurchaseInfo());
        }
        internal static void TrackInAppPurchaseEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackPurchaseEvent(EventType.InApp, GetCampaignImpression(), GetPurchaseInfo());
        }
        internal static void TrackTransaction([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackTransaction(Transaction.CreateIncome(string.Empty), GetBonusesInfo());
        }
        internal static void TrackFirebaseMappingEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackFirebaseMappingEvent("instanceId");
        }
        internal static void TrackMailingStatusMappingEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackMailingStatusMappingEvent("email");
        }
        internal static void TrackImpressionFailEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackImpressionFailEvent(GetCampaignImpression(), "reason");
        }
        internal static void TrackCampaignImpressionEvent([NotNull] AnalyticsTracker analyticsTracker)
        {
            analyticsTracker.TrackCommonEvent(EventType.Impression, GetCampaignImpression());
        }

        [NotNull]
        internal static SubsystemsCollection Create<T>(EventType eventType, Action<object> eventReceivedCallback, [NotNull] out T analyticsTracker, [NotNull] out ServerApi serverApi) where T : IAnalyticsTracker
        {
            var subsystems = new SubsystemsCollection();
            var internalConfigPrefs = InternalConfigPrefs.Create(EditorModeTestsEnvironment.InternalConfigPrefsPath).AddTo(subsystems);
            var platform = new PlatformDefault(internalConfigPrefs).AddTo(subsystems);
            var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
            var prefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, appVersionProvider).AddTo(subsystems);
            var appStatePrefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath).AddTo(subsystems);
            var network = new NetworkMoq((_, message, _) =>
            {
                if (NetworkMoq.TryExtractEvent(message, eventType, out IAnalyticsEvent requestedEvent))
                {
                    eventReceivedCallback?.Invoke(requestedEvent);
                }
                return NetworkMoq.OkResult(message);
            }).AddTo(subsystems);
            var authConfig = new AuthorizationConfigBuilder("AppName", appVersionProvider, platform, prefs, internalConfigPrefs, false).AddTo(subsystems);
            serverApi = new ServerApi(EndpointUrl.Staging, network, prefs, appStatePrefs, authConfig, platform).AddTo(subsystems);
            analyticsTracker = typeof(T) == typeof(MinimalAnalyticsTracker)
                ? (T)(object)new MinimalAnalyticsTracker(serverApi, prefs, appStatePrefs, platform, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(subsystems)
                : (T)(object)new AnalyticsTracker(serverApi, prefs, platform, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(subsystems);
            return subsystems;
        }

        private static ApplovinAdsImpression GetApplovinAdsImpression()
        {
            return new ApplovinAdsImpression
            {
                Id = "id",
                Network = "network",
                ImpressionId = "impressionId",
                Revenue = 0,
                Precision = "precision",
            };
        }

        private static PurchaseInfo GetPurchaseInfo()
        {
            return new PurchaseInfo
            {
                Product = new ProductInfo(null, null, null, store: PurchaseStore.Native),
                TransactionId = "TransactionId",
                OriginalTransactionId = "OriginalTransactionId",
            };
        }

        private static IEnumerable<BonusInfo> GetBonusesInfo()
        {
            return new List<BonusInfo>
            {
                Capacity = 0,
            };
        }

        private static CampaignImpression GetCampaignImpression()
        {
            return new CampaignImpression
            {
                CampaignName = "campaignName",
                IsDefaultConfig = true,
                ImpressionNumber = 0,
                SessionImpressionNumber = 0,
                CampaignTypeImpressionNumber = 0,
                CampaignTypeSessionImpressionNumber = 0,
                EventName = "eventName",
                EventNumber = 0,
                SessionEventNumber = 0,
                SessionNumber = 0,
                ProductId = "id",
                ProductIdCreative = "idCreative",
                Parameters = new Dictionary<string, object>()
                {
                    { "id", "id" },
                },
            };
        }
    }
}