﻿using System.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MinimalMagifyClientTests
    {
        public class Purchases : MinimalMagifyClientTestsPart
        {
            private const string ProductId = "productId";
            private const string Price = "price";
            private const string Currency = "currency";
            private const string TransactionId = "transactionId";
            private const string PurchaseToken = "purchaseToken";
            private const string OriginalTransactionId = "originalTransactionId";
            private const string Receipt = "receipt";

            [Test]
            public async Task TrackExternalInApp_HaveInAppEvent()
            {
                // Arrange
                var got = false;
                var @event = default(ProductPurchaseEvent);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (!got) got = NetworkMoq.TryExtractEvent(message, EventType.InApp, out @event);
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefaultAndInit(Environment.Staging, network);

                // Act
                magifyClient.TrackExternalInAppFor(ProductId, Price, Currency, TransactionId, PurchaseToken, OriginalTransactionId, Receipt, needValidation: true, PurchaseStore.Native);
                await Task.Delay(100);

                // Assert
                got.Should().Be(true);
                @event.Should().NotBeNull();
                @event.ProductId.Should().Be(ProductId);
                @event.Price.Should().Be(Price);
                @event.Currency.Should().Be(Currency);
                @event.TransactionId.Should().Be(TransactionId);
                @event.OriginalTransactionId.Should().Be(OriginalTransactionId);
            }

            [Test]
            [TestCase(true)]
            [TestCase(false)]
            public async Task TrackExternalSubscription_HaveSubscriptionEvent(bool isTrial)
            {
                // Arrange
                var got = false;
                var @event = default(ProductPurchaseEvent);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (!got) got = NetworkMoq.TryExtractEvent(message, isTrial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation, out @event);
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefaultAndInit(Environment.Staging, network);

                // Act
                magifyClient.TrackExternalSubscriptionActivation(isTrial, ProductId, Price, Currency, TransactionId, PurchaseToken, OriginalTransactionId, Receipt, needValidation: true, PurchaseStore.Native);
                await Task.Delay(100);

                // Assert
                got.Should().Be(true);
                @event.Should().NotBeNull();
                @event.ProductId.Should().Be(ProductId);
                @event.Price.Should().Be(Price);
                @event.Currency.Should().Be(Currency);
                @event.TransactionId.Should().Be(TransactionId);
                @event.OriginalTransactionId.Should().Be(OriginalTransactionId);
            }

            [Test]
            public async Task TrackExternalInApp_HaveVerifyRequest()
            {
                // Arrange
                var got = false;
                var verifyPurchasePayload = default(VerifyPurchasePayload);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (!got) got = NetworkMoq.TryExtractVerification(message, out verifyPurchasePayload);
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefaultAndInit(Environment.Staging, network);

                // Act
                magifyClient.TrackExternalInAppFor(ProductId, Price, Currency, TransactionId, PurchaseToken, OriginalTransactionId, Receipt, needValidation: true, PurchaseStore.Native);
                await Task.Delay(100);

                // Assert
                got.Should().Be(true);
                verifyPurchasePayload.Should().NotBeNull();
                verifyPurchasePayload.ProductId.Should().Be(ProductId);
            }

            [Test]
            [TestCase(true)]
            [TestCase(false)]
            public async Task TrackExternalSubscription_HaveVerifyRequest(bool isTrial)
            {
                // Arrange
                var got = false;
                var @event = default(ProductPurchaseEvent);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (!got) got = NetworkMoq.TryExtractEvent(message, isTrial ? EventType.TrialActivation : EventType.PaidSubscriptionActivation, out @event);
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefaultAndInit(Environment.Staging, network);

                // Act
                magifyClient.TrackExternalSubscriptionActivation(isTrial, ProductId, Price, Currency, TransactionId, PurchaseToken, OriginalTransactionId, Receipt, needValidation: true, PurchaseStore.Native);
                await Task.Delay(100);

                // Assert
                got.Should().Be(true);
                @event.Should().NotBeNull();
                @event.ProductId.Should().Be(ProductId);
                @event.Price.Should().Be(Price);
                @event.Currency.Should().Be(Currency);
                @event.TransactionId.Should().Be(TransactionId);
                @event.OriginalTransactionId.Should().Be(OriginalTransactionId);
            }
        }
    }
}