using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using NSubstitute;
using NUnit.Framework;

namespace Magify.Tests
{
    public class EventHandlerTests
    {
        [SetUp]
        public void Setup()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [TearDown]
        public void Cleanup()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [Test]
        public void WhenEventHandlerDontHaveEvent_ThenThereWillBeNoAttemptsToSendEvents()
        {
            // Arrange
            var mockServerApi = Substitute.For<IServerApi>();
            using var _ = CreateAppStatePrefs(out var appStatePrefs);
            var eventHandler = new EventHandler<CustomSessionEvent>(EventType.SessionEvent, mockServerApi, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath);
            eventHandler.Initialize();

            // Act
            var config = new AnalyticsConfiguration { GroupSize = 2 };
            eventHandler.SetupAnalyticsConfig(config);

            // Assert
            mockServerApi
                .DidNotReceive()
                .SendEvents(Arg.Any<EventType>(), Arg.Any<IList<CustomSessionEvent>>(), Arg.Any<CancellationToken>());
        }

        [Test]
        public void WhenHandleOneEvent_AndHasGroupSizeBiggerThenOne_ThenEventShouldNotBeSend()
        {
            // Arrange
            var mockServerApi = Substitute.For<IServerApi>();
            using var _ = CreateAppStatePrefs(out var appStatePrefs);
            var eventHandler = new EventHandler<CustomSessionEvent>(EventType.SessionEvent, mockServerApi, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath);
            eventHandler.Initialize();
            var config = new AnalyticsConfiguration { GroupSize = 2 };
            eventHandler.SetupAnalyticsConfig(config);

            // Act
            eventHandler.HandleEvent(new CustomSessionEvent());

            // Assert
            mockServerApi
                .DidNotReceive()
                .SendEvents(Arg.Any<EventType>(), Arg.Any<IList<CustomSessionEvent>>(), Arg.Any<CancellationToken>());
        }

        [Test]
        public async Task WhenSetSyncIntervalBiggerThanZero_AndHaveSomeEventsToSend_ThenShouldReceivedCallToServer()
        {
            // Arrange
            var mockServerApi = Substitute.For<IServerApi>();
            using var _ = CreateAppStatePrefs(out var appStatePrefs);
            mockServerApi.SendEvents(Arg.Any<EventType>(), Arg.Any<IList<CustomSessionEvent>>(), Arg.Any<CancellationToken>())
                .Returns(UniTask.FromResult(false), UniTask.FromResult(true));
            var eventHandler = new EventHandler<CustomSessionEvent>(EventType.SessionEvent, mockServerApi, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath);
            eventHandler.Initialize();
            eventHandler.SetupAnalyticsConfig(new AnalyticsConfiguration { SyncIntervalInSeconds = 1 });

            // Act
            eventHandler.HandleEvent(new CustomSessionEvent());
            await Task.Delay(300);

            // Assert
            (eventHandler as IDisposable)?.Dispose();
            mockServerApi
                .Received(1)
                .SendEvents(Arg.Any<EventType>(), Arg.Any<IList<CustomSessionEvent>>(), Arg.Any<CancellationToken>())
                .Forget();
        }

        [NotNull]
        private IDisposable CreateAppStatePrefs([NotNull] out AppStatePrefs prefs)
        {
            return prefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath);
        }
    }
}