﻿using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        public class LtoCampaignManagerTests
        {
            [Test]
            [TestCase(ContextKind.Default)]
            public void WhenUpdateContextWithDefaultContextKind_ThenDefaultContentShouldBeNotEmpty(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var _, out var ltoCampaignManager);
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)ltoCampaignManager).UpdateContext(context, kind);

                //Assert
                ltoCampaignManager.DefaultContent.Should()!.NotBeEmpty();
                ltoCampaignManager.DefaultContent.Count.Should()!.Be(context.DefaultContent?.Count);
                ltoCampaignManager.DefaultContent.ShouldBeSameInAnyOrder(context.DefaultContent);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            public void WhenUpdateContextWithDefaultContextKindAndEmptyContext_ThenDefaultContentShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var _, out var ltoCampaignManager);
                var context = new CampaignsContext();

                //Act
                ((IContextListener)ltoCampaignManager).UpdateContext(context, kind);

                //Assert
                ltoCampaignManager.DefaultContent.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var _, out var ltoCampaignManager);
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                IReadOnlyList<DefaultContentItem> firstUpdateContentItems = default;
                IReadOnlyList<DefaultContentItem> secondUpdateContentItems = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)ltoCampaignManager).UpdateContext(firstContext, kind);
                    firstUpdateContentItems = ltoCampaignManager.DefaultContent;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)ltoCampaignManager).UpdateContext(secondContext, kind);
                    secondUpdateContentItems = ltoCampaignManager.DefaultContent;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstUpdateContentItems.Count.Should()!.Be(firstContext.DefaultContent?.Count);
                firstUpdateContentItems.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.DefaultContent?.Select(c => c?.Name).ToArray());

                secondUpdateContentItems.Count.Should()!.Be(secondContext.DefaultContent?.Count);
                secondUpdateContentItems.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.DefaultContent?.Select(c => c?.Name).ToArray());
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenLaunchLtoOffer_ThenActiveLtoOffersShouldBeNotEmpty(ContextKind kind)
            {
                //Arrange
                using var subsystems = CreateSubsystems(out var _, out var ltoCampaignManager);
                var context = CreateCampaignsContext();
                var campaignRecord = GetCampaignRecord();

                //Act
                ltoCampaignManager.TryToLaunchOffer(campaignRecord, new Trigger());
                subsystems.UpdateContextForAll(context, kind);

                //Assert
                ltoCampaignManager.GetActiveLtoOffers().Should()!.NotBeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenCompleteLtoOffer_ThenActiveLtoOffersShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                using var subsystems = CreateSubsystems(out var _, out var ltoCampaignManager);
                var context = CreateCampaignsContext();
                var campaignRecord = GetCampaignRecord();

                //Act
                ltoCampaignManager.TryToLaunchOffer(campaignRecord, new Trigger());
                subsystems.UpdateContextForAll(context, kind);
                ltoCampaignManager.CompleteOffer("campaignModelName");

                //Assert
                ltoCampaignManager.GetActiveLtoOffers().Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Saved, 10)]
            [TestCase(ContextKind.Downloaded, 10)]
            [TestCase(ContextKind.Saved, 50)]
            [TestCase(ContextKind.Downloaded, 50)]
            [TestCase(ContextKind.Saved, 100)]
            [TestCase(ContextKind.Downloaded, 100)]
            public void WhenLaunchSameLtoOfferMultiplyTimes_ThenActiveLtoOffersShouldBeOne(ContextKind kind, int iterateCount)
            {
                //Arrange
                using var subsystems = CreateSubsystems(out var _, out var ltoCampaignManager);
                var context = CreateCampaignsContext();
                var campaignRecord = GetCampaignRecord();

                //Act
                for (int i = 0; i < iterateCount; i++)
                {
                    ltoCampaignManager.TryToLaunchOffer(campaignRecord, new Trigger());
                }
                subsystems.UpdateContextForAll(context, kind);

                //Assert
                ltoCampaignManager.GetActiveLtoOffers().Count.Should()!.Be(1);
            }
        }
    }
}