﻿using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class DefaultProductsContainerTests
        {
            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenDefaultProductsShouldBeSameDefaultProductsCount(ContextKind kind)
            {
                //Arrange
                var defaultProductsContainer = new Magify.DefaultProductsContainer();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)defaultProductsContainer).UpdateContext(context, kind);

                //Assert
                defaultProductsContainer.DefaultProducts!.Count.Should()!.Be(context.DefaultProducts!.Count);
                defaultProductsContainer.DefaultProducts!.ShouldBeSameInAnyOrder(context.DefaultProducts);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenResetDefaultProducts_ThenDefaultProductsShouldBeNull(ContextKind kind)
            {
                //Arrange
                var defaultProductsContainer = new Magify.DefaultProductsContainer();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)defaultProductsContainer).UpdateContext(context, kind);
                defaultProductsContainer.ResetDefaultProducts();

                //Assert
                defaultProductsContainer.DefaultProducts!.Should()!.BeNull();
                defaultProductsContainer.InAppProducts!.Should()!.BeEmpty();
                defaultProductsContainer.SubscriptionProducts!.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWithEmptyDefaultProducts_ThenDefaultProductsShouldBeNull(ContextKind kind)
            {
                //Arrange
                var defaultProductsContainer = new Magify.DefaultProductsContainer();
                var context = new CampaignsContext();

                //Act
                ((IContextListener)defaultProductsContainer).UpdateContext(context, kind);

                //Assert
                defaultProductsContainer.DefaultProducts!.Should()!.BeNull();
            }

            [Test]
            [TestCase(ContextKind.Default, false)]
            [TestCase(ContextKind.Saved, true)]
            [TestCase(ContextKind.Downloaded, true)]
            public void WhenUpdateContextWhenDefaultProductsAlreadyExist_ThenDefaultProductsShouldBeNew(ContextKind kind, bool expectedEqual)
            {
                //Arrange
                var defaultProductsContainer = new DefaultProductsContainer();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)defaultProductsContainer).UpdateContext(firstContext, kind);
                var firstDefaultProducts = defaultProductsContainer.DefaultProducts;
                ((IContextListener)defaultProductsContainer).UpdateContext(secondContext, kind);
                var secondDefaultProducts = defaultProductsContainer.DefaultProducts;

                //Assert
                if (expectedEqual)
                {
                    firstDefaultProducts.Should()!.NotEqual(secondDefaultProducts);
                }
                else
                {
                    firstDefaultProducts.Should()!.Equal(secondDefaultProducts);
                }
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateDefaultProductsShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                var listener = new CampaignsCollection();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                IReadOnlyList<DefaultProduct> firstUpdateDefaultProducts = default;
                IReadOnlyList<DefaultProduct> secondUpdateDefaultProducts = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(firstContext, kind);
                    firstUpdateDefaultProducts = firstContext.DefaultProducts;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(secondContext, kind);
                    secondUpdateDefaultProducts = secondContext.DefaultProducts;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstUpdateDefaultProducts.Count.Should()!.Be(firstContext.DefaultProducts?.Count);
                firstUpdateDefaultProducts.ShouldBeSameInAnyOrder(firstContext.DefaultProducts);

                secondUpdateDefaultProducts.Count.Should()!.Be(secondContext.DefaultProducts?.Count);
                secondUpdateDefaultProducts.ShouldBeSameInAnyOrder(secondContext.DefaultProducts);
            }
        }
    }
}