using System;
using JetBrains.Annotations;
using Newtonsoft.Json;
using Newtonsoft.Json.Converters;
using UnityEngine.Scripting;

namespace Magify
{
    public enum TimeUnit
    {
        Ticks,
        Millisecond,
        Second,
        Minute,
        Hour,
        Day
    }

    [Preserve]
    public class UnitDateTimeConverter : DateTimeConverterBase
    {
        public TimeUnit TimeUnit { get; set; }
        public bool SupportFloatingNumbers { get; set; }

        [UsedImplicitly, Preserve]
        public UnitDateTimeConverter() : this(TimeUnit.Ticks, false)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitDateTimeConverter(TimeUnit timeUnit) : this(timeUnit, false)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitDateTimeConverter(bool supportFloatingNumbers) : this(TimeUnit.Ticks, supportFloatingNumbers)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitDateTimeConverter(TimeUnit timeUnit, bool supportFloatingNumbers)
        {
            TimeUnit = timeUnit;
            SupportFloatingNumbers = supportFloatingNumbers;
        }

        public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
        {
            if (value is DateTime dateTime)
            {
                var span = dateTime - DateTime.UnixEpoch;
                writer.WriteValue(span, TimeUnit, SupportFloatingNumbers);
            }
            else
            {
                writer.WriteNull();
            }
        }

        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            var timeSpan = reader.ReadTimeSpan(TimeUnit, SupportFloatingNumbers);
            if (timeSpan == null)
            {
                var nullable = objectType.IsGenericType && objectType.GetGenericTypeDefinition() == typeof(Nullable<>);
                return nullable ? null : throw new JsonSerializationException($"Cannot convert {reader.Value} value to {objectType.Name}.");
            }
            var value = timeSpan.Value;
            return DateTime.UnixEpoch + value;
        }
    }
}