﻿using System;
using System.Buffers;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public sealed partial class BinaryStorage : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        [NotNull]
        private readonly string _storageFilePath;
        [NotNull]
        private readonly string _storageFilePathTmp;
        [NotNull, ItemNotNull]
        private readonly List<Section> _sections;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly object _lockFile = new();

        private ChangeScope _multipleChangeScope;

        private BinaryStorage([NotNull] string storageFilePath, [NotNull, ItemNotNull] List<Section> sections)
        {
            _storageFilePath = storageFilePath;
            _storageFilePathTmp = storageFilePath + ".tmp";
            _sections = sections;
            foreach (var section in sections)
            {
                if (section is IDisposable disposable)
                {
                    disposable.AddTo(_disposables);
                }
                section.OnChanged += () =>
                {
                    if (_multipleChangeScope == null) SaveDataOnDisk();
                };
            }
        }

        public bool Disposed { get; private set; }

        public bool Supports<T>()
        {
            ThrowIfDisposed();
            return _sections.FirstOrDefault(c => c is TypedSection<T>) != null;
        }

        public bool Has<T>([NotNull] string key)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            return section.Has(key);
        }

        public T Get<T>([NotNull] string key, T initValue = default)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            return section.Get(key, initValue);
        }

        [NotNull]
        public IReactiveProperty<T> GetReactiveProperty<T>([NotNull] string key, T initValue = default)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            return section.GetReactiveProperty(key, initValue);
        }

        public bool Set<T>([NotNull] string key, T value)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            var changed = section.Set(key, value);
            return changed;
        }

        public bool Remove<T>([NotNull] string key)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            return section.Remove(key);
        }

        public int Remove<T>([NotNull] Func<string, bool> predicate)
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            using (MultipleChangeScope())
            {
                return section.Remove(predicate);
            }
        }

        public void ResetOnly<T>()
        {
            ThrowIfDisposed();
            var section = GetTypedSection<T>();
            if (!section.HasData) return;
            section.Reset();
        }

        public IList<T> GetList<T>([NotNull] string key)
        {
            ThrowIfDisposed();
            return GetCollectionSection<T, ReactiveList<T>>().Get(key);
        }

        [NotNull, ItemNotNull]
        public ISet<T> GetSet<T>([NotNull] string key)
        {
            ThrowIfDisposed();
            return GetCollectionSection<T, ReactiveHashSet<T>>().Get(key);
        }

        [NotNull]
        public IDictionary<TKey, TValue> GetDictionary<TKey, TValue>([NotNull] string key)
        {
            ThrowIfDisposed();
            return GetCollectionSection<KeyValuePair<TKey, TValue>, ReactiveDictionary<TKey, TValue>>().Get(key);
        }

        public void ResetAll()
        {
            ThrowIfDisposed();
            using (MultipleChangeScope())
            {
                foreach (var section in _sections)
                {
                    section.Reset();
                }
            }
        }

        [NotNull]
        public IDisposable MultipleChangeScope()
        {
            ThrowIfDisposed();
            if (_multipleChangeScope != null)
            {
                _multipleChangeScope.Inc();
                return _multipleChangeScope;
            }
            _multipleChangeScope = new ChangeScope(this);
            return _multipleChangeScope;
        }

        [NotNull]
        private TypedSection<T> GetTypedSection<T>()
        {
            if (_sections.FirstOrDefault(c => c is TypedSection<T>) is not TypedSection<T> section)
            {
                throw new BinaryStorageDoesntSupportTypeException(Path.GetFileName(_storageFilePath), typeof(T));
            }
            return section;
        }

        [NotNull]
        private CollectionSection<T, TCollection> GetCollectionSection<T, TCollection>()
            where TCollection : class, IReactiveCollection, IRefillableCollection<T>, ICollection<T>, new()
        {
            if (_sections.FirstOrDefault(c => c is CollectionSection<T, TCollection>) is not CollectionSection<T, TCollection> section)
            {
                throw new BinaryStorageDoesntSupportCollectionException(_storageFilePath, typeof(TCollection), typeof(T));
            }
            return section;
        }

        internal void SaveDataOnDisk()
        {
            ThrowIfDisposed();
            _multipleChangeScope = null;
            if (_sections.All(c => !c.HasData))
            {
                return;
            }
            if (!TrySaveDataOnDisk(logException: false))
                if (!TrySaveDataOnDisk(logException: false))
                    TrySaveDataOnDisk(logException: true);
            SaveAsJsonForDebug(_storageFilePath, _sections);
        }

        private bool TrySaveDataOnDisk(bool logException)
        {
            lock (_lockFile)
            {
                try
                {
                    var directoryName = Path.GetDirectoryName(_storageFilePathTmp);
                    if (string.IsNullOrEmpty(directoryName))
                    {
                        throw new ArgumentNullException(directoryName);
                    }
                    CreateDirectoryWithRetries(directoryName, maxRetries: 3);
                    using (var stream = new FileStream(_storageFilePathTmp, FileMode.Create))
                    {
                        using (var writer = new BinaryWriter(stream, Encoding.UTF8))
                        {
                            foreach (var section in _sections.Where(section => section.HasData))
                            {
                                writer.Write(section!.Name); //section cannot be null, if it happened there must be exception
                                section.WriteTo(writer);
                            }
                        }
                    }
                    if (File.Exists(_storageFilePath))
                    {
                        File.Delete(_storageFilePath);
                    }
                    File.Move(_storageFilePathTmp, _storageFilePath);
                    return true;
                }
                catch (Exception e)
                {
                    // ReSharper disable ConditionIsAlwaysTrueOrFalse
                    if (logException)
                        _logger.LogException(new MagifyFailedToSaveBinaryStorageException(_storageFilePath, e));
                    else _logger.LogWarning(new MagifyFailedToSaveBinaryStorageException(_storageFilePath, e).ToString());
                    // ReSharper restore ConditionIsAlwaysTrueOrFalse
                    return false;
                }
            }
        }

        private static void CreateDirectoryWithRetries([NotNull] string directoryName, int maxRetries)
        {
            var retryNumber = 0;
            while (!Directory.Exists(directoryName) && retryNumber++ < maxRetries)
            {
                Directory.CreateDirectory(directoryName);
            }
        }

        internal void LoadDataFromDisk()
        {
            if (!tryLoadData(false))
                tryLoadData(true);
            return;

            bool tryLoadData(bool isRetry)
            {
                lock (_lockFile)
                {
                    try
                    {
                        LoadDataFromDisk(Encoding.UTF8, false);
                        return true;
                    }
                    catch (NoBinaryStorageSectionForTypeException)
                    {
                        throw;
                    }
                    catch (Exception e)
                    {
                        _logger.LogException(new MagifyFailedToLoadBinaryStorageException(_storageFilePath, isRetry, e));
                        return false;
                    }
                }
            }
        }

        internal void LoadDataFromDisk([NotNull] Encoding encoding, bool throwOnError)
        {
            ThrowIfDisposed();
            if (!File.Exists(_storageFilePath))
            {
                return;
            }
            using var stream = new FileStream(_storageFilePath, FileMode.Open);
            using var reader = new BinaryReader(stream, encoding);
            while (stream.Position != stream.Length)
            {
                var typeName = reader.ReadString();
                var section = _sections.FirstOrDefault(c => c.Name == typeName);

                if (section != null)
                {
                    section.ReadFrom(reader);
                }
                else if (throwOnError)
                {
                    throw new NoBinaryStorageSectionForTypeException(typeName);
                }
                else
                {
                    // null section means that this section was saved with old binary type
                    // it is definitely not good but all we can do here is just read and skip it
                    // all sections has specific format, and it is possible to just read data and ignore it
                    // [count: int] - amount of items in section
                    // [key: string][size: int][data: bytes] - data for each entry in section
                    var count = reader.ReadInt32();
                    for (var i = 0; i < count; i++)
                    {
                        reader.ReadString(); // ignore [key]
                        var size = reader.ReadInt32();
                        var buffer = ArrayPool<byte>.Shared?.Rent(size) ?? new byte[size];
                        // ReSharper disable once MustUseReturnValue
                        stream.Read(buffer, 0, size);
                        ArrayPool<byte>.Shared?.Return(buffer);
                    }
                }
            }
        }

        /// <returns>
        /// Base64 encoded content of the file.
        /// </returns>
        [NotNull]
        internal string GetFileContent()
        {
            ThrowIfDisposed();
            if (File.Exists(_storageFilePath))
            {
                byte[] fileBytes;
                using (var stream = new FileStream(_storageFilePath, FileMode.Open, FileAccess.Read))
                {
                    fileBytes = new byte[stream.Length];
                    _ = stream.Read(fileBytes, 0, fileBytes.Length);
                }
                return Convert.ToBase64String(fileBytes);
            }
            return string.Empty;
        }

        /// <summary>
        /// Rewrites content of the file, and then updates all reactive properties and collections.
        /// </summary>
        /// <param name="contentBase64">
        /// Base64 encoded content of the file.
        /// </param>
        internal void RewriteContent([NotNull] string contentBase64)
        {
            ThrowIfDisposed();
            // We have to finish all writing operations to the storage, it's okay, because all data will be overwritten
            _multipleChangeScope?.Dispose();
            lock (_lockFile)
            {
                if (File.Exists(_storageFilePathTmp))
                {
                    File.Delete(_storageFilePathTmp);
                }
                var directoryName = Path.GetDirectoryName(_storageFilePathTmp);
                if (!string.IsNullOrEmpty(directoryName) && !Directory.Exists(directoryName))
                {
                    Directory.CreateDirectory(directoryName);
                }

                var bytes = Convert.FromBase64String(contentBase64);
                using (var stream = new FileStream(_storageFilePathTmp, FileMode.Create))
                {
                    stream.Write(bytes, 0, bytes.Length);
                }
                if (File.Exists(_storageFilePath))
                {
                    File.Delete(_storageFilePath);
                }
                File.Move(_storageFilePathTmp, _storageFilePath);
            }

            // refresh all reactive properties and collections
            using var _ = MultipleChangeScope();
            _sections.Where(s => s is ICollectionSection).ForEach(s => s?.Reset());
            LoadDataFromDisk(Encoding.UTF8, true);
            SaveAsJsonForDebug(_storageFilePath, _sections);
        }

        private void ThrowIfDisposed()
        {
            if (Disposed)
            {
                throw new BinaryStorageAlreadyDisposedException(_storageFilePath);
            }
        }

        public void Dispose()
        {
            if (Disposed) return;
            if (_multipleChangeScope != null)
            {
                throw new CannotDisposeBinaryStorageBeforeMultipleChangeScopeException(_storageFilePath);
            }
            _disposables.Release();
            _sections.Clear();
            Disposed = true;
            UnlockFilePathInEditor(_storageFilePath);
        }
    }
}