using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        public UniTask<PurchaseVerificationResult> VerifyPurchase([NotNull] PurchaseRecord purchaseRecord, bool waitForResult, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                WebRequestMethods.VerifyPurchase(_platform.RuntimePlatform),
                createVerifyPurchaseRequestPayload,
                useWebRequestWithRepeats: false,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestVerifyPurchaseContext,
                in config,
                cancellationToken);
            return request.HandleByDefault(config, rejectedResult: PurchaseVerificationResult.Fail(purchaseRecord));

            WebRequestMessage createVerifyPurchaseRequestPayload()
            {
                ThrowIfDisposed();
                return new WebRequestMessage
                {
                    Method = WebRequestMethods.VerifyPurchase(_platform.RuntimePlatform),
                    Payload = new VerifyPurchasePayload(purchaseRecord, waitForResult),
                };
            }
        }

        private RepeatState VerifyPurchaseResponseHandler(WebResponseMessage response, out PurchaseVerificationResult result)
        {
            if (response.Result == UnityWebRequest.Result.Success)
            {
                _logger.Log($"Purchase verified successfully. Product id: {getPayload().ProductId}");
                result = PurchaseVerificationResult.Success(getPayload());
                return RepeatState.Finish;
            }

            _logger.Log($"Purchase verification failed. Product id: {getPayload().ProductId}. \nResponse: {JsonFacade.SerializeObject(response)}");
            PurchaseVerificationResult? temp = null;
            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = PurchaseVerificationResult.Fail(getPayload());
                    return repeatState;
                }
                switch (errorResponse?.Error?.Code)
                {
                    case ErrorCode.AppSupportError:
                        _logger.Log("The application is not configured in the purchase validator. The purchase cannot be validated.");
                        temp = PurchaseVerificationResult.DoesntSupport(getPayload());
                        break;

                    // Works only for 'with_status' (waitForResult) flag
                    case ErrorCode.InvalidPurchase:
                        _logger.Log($"Purchase is invalid. Purchase: {JsonFacade.SerializeObject(getPayload())}");
                        temp = PurchaseVerificationResult.Invalid(getPayload());
                        break;

                    case ErrorCode.InvalidGoogleCredentials:
                        _logger.Log($"Google credentials are invalid. Purchase: {JsonFacade.SerializeObject(getPayload())}");
                        temp = PurchaseVerificationResult.InvalidGoogleCredentials(getPayload());
                        break;
                }
            }

            result = temp ?? PurchaseVerificationResult.Fail(getPayload());
            return RepeatState.Finish;

            [NotNull]
            VerifyPurchasePayload getPayload() => (VerifyPurchasePayload)response.RequestMessage.Payload!;
        }

        public void CancelAllVerifyPurchaseRequests()
        {
            _logger.Log($"{nameof(CancelAllVerifyPurchaseRequests)} has been called");
            _requestVerifyPurchaseContext.CancelAllRequests();
        }
    }
}