using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using UnityEngine;
using UnityEngine.Pool;

namespace Magify
{
    internal class LtoCampaignManager : ILegacyMigrator, IInitializable, IContextListener, IForegroundListener, IDisposable
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Lto);

        private readonly CampaignsTracker _campaignsTracker;
        private readonly CampaignsCollection _campaignsCollection;
        private readonly GeneralPrefs _generalPrefs;
        private readonly PlatformAPI _platform;
        private readonly Counters _counters;
        private readonly OffersPrefs _offersPrefs;
        private readonly OffersStorage _ltoStorage;
        [NotNull]
        private readonly AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly OffersCollection _activeOffers = new();
        [NotNull]
        private readonly object _lock = new();

        public event Action<LtoInfo> OnAdded;
        public event Action<LtoInfo> OnUpdated;
        public event Action<LtoInfo> OnRemoved;
        public event Action<LtoInfo> OnFinished;

        [NotNull]
        private IReadOnlyList<DefaultContentItem> _defaultContent = ArraySegment<DefaultContentItem>.Empty;
        [NotNull]
        internal IReadOnlyList<DefaultContentItem> DefaultContent => _defaultContent;

        public ConfigScope SuitableScope => ConfigScope.Campaigns;

        public LtoCampaignManager(
            CampaignsTracker campaignsTracker,
            CampaignsCollection campaignsCollection,
            GeneralPrefs generalPrefs,
            OffersPrefs offersPrefs,
            PlatformAPI platform,
            Counters counters,
            string storagePath,
            AppVersionProvider appVersionProvider)
        {
            _ltoStorage = new OffersStorage(storagePath);
            _campaignsTracker = campaignsTracker;
            _campaignsCollection = campaignsCollection;
            _generalPrefs = generalPrefs;
            _platform = platform;
            _counters = counters;
            _offersPrefs = offersPrefs;
            _appVersionProvider = appVersionProvider;
        }

        void ILegacyMigrator.Migrate(MigrationData data)
        {
            _logger.Log("Try to migrate limited offers from native.");
            if (_ltoStorage.Exists())
            {
                _logger.Log($"Saved offers already exists in folder {_ltoStorage.RootFolderPath}. Skip migration.");
                return;
            }
            if (data.ActiveTimeLimitedOffers is not { Count: > 0 })
            {
                _logger.Log("No LTO in migration data. Skip migration.");
                return;
            }
            _logger.Log($"Migrate limited offers ({data.ActiveTimeLimitedOffers.Count}) from native.");
            _ltoStorage.SaveActiveOffers(data.ActiveTimeLimitedOffers);
            _logger.Log($"Migrate limited offers completed to folder {_ltoStorage.RootFolderPath}.");
        }

        void IInitializable.Initialize()
        {
            _logger.Log("Initialize limited offers manager.");
            _activeOffers.AddTo(_disposables);
            _generalPrefs.SubscriptionStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => CompleteActiveOffers(offer =>
                    offer.Campaign.SubscriptionStatus?.IsCorresponded(_generalPrefs.SubscriptionStatus.Value) == false ||
                    offer.Campaign.PayingStatus?.IsCorresponded(_generalPrefs.SubscriptionStatus.Value, _generalPrefs.InAppStatus.Value) == false))
                .AddTo(_disposables);

            _generalPrefs.InAppStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => CompleteActiveOffers(offer =>
                    offer.Campaign.InAppStatus?.IsCorresponded(_generalPrefs.InAppStatus.Value) == false ||
                    offer.Campaign.PayingStatus?.IsCorresponded(_generalPrefs.SubscriptionStatus.Value, _generalPrefs.InAppStatus.Value) == false))
                .AddTo(_disposables);

            _generalPrefs.AuthorizationStatus
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => CompleteActiveOffers(offer => offer.Campaign.AuthorizationStatus?.IsCorresponded(_generalPrefs.AuthorizationStatus.Value) == false))
                .AddTo(_disposables);

            _platform.OnNewPackageInstalled
                .Subscribe(OnNewPackageInstalled)
                .AddTo(_disposables);

            LoadExistingOffers();
        }

        void IForegroundListener.OnForeground()
        {
            CompleteActiveOffers(offer => ShouldBeCompleted(offer, out _));
        }

        private void LoadExistingOffers()
        {
            _logger.Log("Load offers from previous session");
            var offers = _ltoStorage.LoadActiveOffers(_campaignsCollection);
            if (offers == null)
            {
                _logger.Log("No saved offers found");
                return;
            }

            _logger.Log($"Saved offers count is {offers.Count}");
            var completed = ListPool<LtoModel>.Get();
            foreach (var offer in offers)
            {
                if (ShouldBeCompleted(offer, out var completeReason))
                {
                    _logger.Log($"Offer '{offer.CampaignName}' will be completed by reason: {completeReason}");
                    completed.Add(offer);
                }
                else
                {
                    _logger.Log($"Offer '{offer.CampaignName}' will be added");
                    AddActiveOffer(offer);
                }
            }

            if (completed.Count != 0)
            {
                foreach (var offer in completed)
                {
                    CompleteFinishedOffer(offer);
                }

                SaveActiveOffers();
            }

            ListPool<LtoModel>.Release(completed);
        }

        public void Dispose()
        {
            _disposables.Release();
        }

        void IContextListener.UpdateContext([NotNull] CampaignsContext context, ContextKind kind)
        {
            lock (_lock)
            {
                switch (kind)
                {
                    case ContextKind.Default when context.DefaultContent != null:
                        _defaultContent = context.DefaultContent;
                        break;
                    case ContextKind.Saved or ContextKind.Downloaded when context.CampaignModels != null:
                        CheckActiveOfferCampaignUpdates(_campaignsCollection.CurrentCampaigns);
                        break;
                }
            }
        }

        [NotNull]
        public IReadOnlyCollection<LtoInfo> GetActiveLtoOffers()
        {
            return _activeOffers.Offers;
        }

        public void CompleteOffer(string name)
        {
            CompleteActiveOffers(offer => offer.CampaignName == name);
        }

        public bool HasActiveOffer(string name)
        {
            return _activeOffers.Contains(name);
        }

        public bool HasActiveOfferBySpot(string name, string spot)
        {
            return FindActiveOfferBySpot(spot)?.Name == name;
        }

        [CanBeNull]
        public CampaignRecord FindActiveOfferBySpot(string spot)
        {
            return _activeOffers.FirstOrDefault(o => o.Spot == spot)?.Record;
        }

        public int GetActivationsPerPeriod(CounterKey key, TimeSpan period)
        {
            var prevStartTime = _offersPrefs.GetLastOfferStartTime(key.ToString());
            if (prevStartTime != default && prevStartTime + period > DateTime.UtcNow)
            {
                return _counters.LtoStart[LtoStartCounterScope.Period, key];
            }
            return 0;
        }

        public void ResetContext()
        {
            _defaultContent = new List<DefaultContentItem>(0);
        }

        private void OnNewPackageInstalled(string packageName)
        {
            _logger.Log($"[LTO] {nameof(OnNewPackageInstalled)}, packageName: {packageName}");
            CompleteActiveOffers(o => !o.Campaign.IsCampaignRelevant(_platform));
        }

        public void OnAppStateChanged()
        {
            _logger.Log($"[LTO] {nameof(OnAppStateChanged)} handler");
            CompleteActiveOffers(offer => ShouldBeCompleted(offer, out _));
        }

        public void OnProductPurchase(string productId)
        {
            _logger.Log($"[LTO] {nameof(OnProductPurchase)}, product: {productId}");
            CompleteActiveOffers(shouldBeCompleted);
            return;

            bool shouldBeCompleted(LtoModel offer)
            {
                var stopOnProductPurchase = offer.Campaign.StopOfferOnPurchase == true;
                return stopOnProductPurchase && offer.Campaign.TryGetNested(productId, out _) ||
                       !offer.Info.PurchaseLimits.IsCorresponded(_generalPrefs.AllUsedProducts);
            }
        }

        public void OnRewardGranted(string productId)
        {
            _logger.Log($"[LTO] {nameof(OnRewardGranted)}, product: {productId}");
            CompleteActiveOffers(shouldBeCompleted);
            return;

            bool shouldBeCompleted(LtoModel offer)
            {
                if (!offer.Campaign.TryGetNested(productId, out var nestedCampaign))
                {
                    return false;
                }

                var nestedLimit = nestedCampaign.RewardOverAfterWatchLimit;
                var isNestedLimitReached = nestedLimit is > 0 && _counters.Nested[NestedCounterScope.Activation, CounterKey.GetKey(nestedName: nestedCampaign.Name)] >= nestedLimit;

                var rewardWatchLimit = offer.Campaign.GetStopRewardAfterWatchLimit();
                var isActivationLimitReached = rewardWatchLimit is > 0 && _counters.Rewards[RewardsCounterScope.Activation, CounterKey.GetKey(campaignName: offer.CampaignName)] >= rewardWatchLimit;

                var isCampaignLimitReached = !offer.Info.IsWithinRewardsGrantLimits(_counters.Rewards);
                return isNestedLimitReached || isActivationLimitReached || isCampaignLimitReached;
            }
        }

        public void OnBonusGranted(string productId)
        {
            CompleteActiveOffers(offer => offer.Campaign.Type == CampaignType.LtoBonus && offer.Campaign.TryGetNested(productId, out _));
        }

        public bool TryToLaunchOffer(CampaignRecord record, Trigger trigger)
        {
            var campaign = record.Campaign;
            _logger.Log($"Try to launch offer: {campaign.Name}");

            var priorityLaunch = trigger.IsPriorityLaunch ?? false;
            var targetSpot = FindSuitableSpotForOffer(campaign, priorityLaunch);
            if (targetSpot == null)
            {
                _logger.Log("LTO launch canceled: no available spots.");
                return false;
            }

            var ltoTag = campaign.LtoTag?.Name;
            if (HaveActiveOfferWithSameTag(ltoTag, targetSpot.Name))
            {
                _logger.Log($"LTO launch canceled: has active offer with same tag:${ltoTag}");
                return false;
            }

            if (!campaign.IsWithinLaunchTimeLimits())
            {
                _logger.Log("LTO launch canceled: invalid lto launch time.");
                return false;
            }

            if (!campaign.IsWithinActivationLimits(_counters.LtoStart))
            {
                _logger.Log("LTO launch canceled: on activation limit.");
                return false;
            }

            if (!campaign.IsWithinAppVersionLimits(_appVersionProvider.AppVersion))
            {
                _logger.Log($"LTO launch canceled: on app version limits. Current version={_appVersionProvider.AppVersion}");
                return false;
            }

            if (!campaign.IsLaunchTimeIntervalPassed(_offersPrefs.GetLastOfferEndTime(campaign.Name)))
            {
                _logger.Log("LTO launch canceled: on activation interval.");
                return false;
            }

            if (!campaign.IsTimeWithinActivationPeriod())
            {
                _logger.Log("LTO launch canceled: outside activation period.");
                return false;
            }

            if (!campaign.IsWithinActivationsPerPeriodLimit(_offersPrefs.GetLastOfferStartTime(campaign.Name), _counters.LtoStart))
            {
                _logger.Log("LTO launch canceled: on activations limit per period.");
                return false;
            }

            ResetActivationValuesPerPeriod(campaign, trigger);
            StartLto(record, targetSpot.Name, trigger.Id, priorityLaunch);

            return true;
        }

        [CanBeNull]
        private Spot FindSuitableSpotForOffer(CampaignModel campaign, bool priorityLaunch)
        {
            if (campaign.Spots == null)
            {
                return null;
            }

            var occupiedSpots = _activeOffers.Offers.Select(i => i.Spot).ToHashSet();
            var freeSpot = campaign.Spots.FirstOrDefault(s => !occupiedSpots.Contains(s.Name));

            if (!priorityLaunch || freeSpot != null)
            {
                return freeSpot;
            }

            return campaign.Spots.FirstOrDefault(spot => _activeOffers.Offers.Any(i => i.Spot == spot.Name && i.HasPriority == false));
        }

        private bool HaveActiveOfferWithSameTag(string tag, string spot)
        {
            foreach (var offer in _activeOffers)
            {
                if (offer.Spot == spot)
                {
                    continue;
                }
                if (offer.Campaign.LtoTag?.Name == tag)
                {
                    return true;
                }
            }
            return false;
        }

        private void ResetActivationValuesPerPeriod(CampaignModel campaign, Trigger trigger)
        {
            var prevStartTime = _offersPrefs.GetLastOfferStartTime(campaign.Name);
            var activationPeriodStartTime = campaign.ActivationPeriod?.GetStartTime();
            if (prevStartTime != default && prevStartTime < activationPeriodStartTime)
            {
                // Reset lto start counter for new activation period
                _campaignsTracker.ResetLtoStartCounter(LtoStartCounterScope.Period, CounterKey.GetKey(campaignName: campaign.Name));
            }

            var keyCampaignPerTrigger = CounterKey.GetKeyCampaignBySource(campaign.Name, trigger.Id);
            var prevStartTimeByTrigger = _offersPrefs.GetLastOfferStartTime(keyCampaignPerTrigger.ToString());
            var currentTime = DateTime.UtcNow;

            if (prevStartTimeByTrigger != default && trigger.ActivationPeriod.HasValue && currentTime > prevStartTimeByTrigger + trigger.ActivationPeriod)
            {
                // Reset activation prefs when period expired
                _campaignsTracker.ResetLtoStartCounter(LtoStartCounterScope.Period, keyCampaignPerTrigger);
                _offersPrefs.ResetLastOfferStartTime(keyCampaignPerTrigger.ToString());
            }
        }

        private void CheckActiveOfferCampaignUpdates(IReadOnlyCollection<CampaignRecord> campaigns)
        {
            var hasChanges = false;
            var completed = ListPool<LtoModel>.Get();
            var active = ListPool<LtoModel>.Get();
            active.AddRange(_activeOffers.Offers);

            foreach (var offer in active)
            {
                var record = campaigns.FirstOrDefault(c => c.Name == offer.CampaignName);
                if (record == null || !record.Campaign.CanBeUpdatedWhileActive())
                {
                    continue;
                }
                var campaign = record.Campaign;
                var trigger = campaign.Triggers?.FirstOrDefault(t => t.Id == offer.Trigger);
                var updatedEndTime = campaign.GetOfferEndTime(offer.StartTime);

                var isPriorityChanged = offer.HasPriority != trigger?.IsPriorityLaunch;
                var isBadgeChanged = offer.Campaign.LtoBadge?.Info.Equals(campaign.LtoBadge?.Info) is false;
                var isEndTimeChanged = offer.EndTime != updatedEndTime;

                if (!isPriorityChanged && !isBadgeChanged && !isEndTimeChanged)
                {
                    continue;
                }

                offer.EndTime = updatedEndTime;
                offer.HasPriority = trigger?.IsPriorityLaunch is true;
                offer.EndTime = updatedEndTime;
                offer.Record = record;
                offer.RebuildBadgeCreative();

                hasChanges = true;
                if (ShouldBeCompleted(offer, out var reason))
                {
                    _logger.Log($"LTO '{offer.CampaignName}' will be completed by reason: {reason}");
                    completed.Add(offer);
                    RemoveActiveOffer(offer);
                }
                else
                {
                    _logger.Log($"LTO '{offer.CampaignName}' will be updated");
                    UpdateActiveOffer(offer, record);
                }
            }

            if (hasChanges)
            {
                SaveActiveOffers();
            }

            foreach (var offer in completed)
            {
                CompleteFinishedOffer(offer);
            }

            ListPool<LtoModel>.Release(completed);
            ListPool<LtoModel>.Release(active);
        }

        private void CompleteActiveOffers(Func<LtoModel, bool> predicate)
        {
            var completed = ListPool<LtoModel>.Get();
            var offers = ListPool<LtoModel>.Get();
            offers.AddRange(_activeOffers.Offers.Where(predicate));
            foreach (var offer in offers)
            {
                _logger.Log($"Complete lto: {offer.CampaignName}");
                RemoveActiveOffer(offer);
                completed.Add(offer);
            }
            foreach (var offer in completed)
            {
                CompleteFinishedOffer(offer);
            }
            if (completed.Count > 0)
            {
                SaveActiveOffers();
            }

            ListPool<LtoModel>.Release(completed);
            ListPool<LtoModel>.Release(offers);
        }

        private void StartLto(CampaignRecord record, string spot, string trigger, bool hasPriority)
        {
            var campaign = record.Campaign;
            _logger.Log($"Start Lto name=${campaign.Name} spot={spot} trigger={trigger} priority={hasPriority}");

            LtoModel conflictedOffer = null;
            if (hasPriority)
            {
                conflictedOffer = _activeOffers.GetModelBySpot(spot);
                if (conflictedOffer != null)
                {
                    _logger.Log($"Complete Lto by priority started offer={record.Name}");
                    RemoveActiveOffer(conflictedOffer);
                }
            }

            var currentTime = DateTime.UtcNow;
            var offer = new LtoModel
            {
                CampaignName = campaign.Name,
                Spot = spot,
                StartTime = currentTime,
                EndTime = campaign.GetOfferEndTime(currentTime),
                HasPriority = hasPriority,
                Trigger = trigger,
                BadgePlaceholder = _defaultContent.FirstOrDefault(i => i.Name == spot)?.Filename,
                Record = record,
            };
            offer.RebuildBadgeCreative();

            _campaignsTracker.ResetActivationCounters(offer);

            var keyCampaignPerTrigger = CounterKey.GetKeyCampaignBySource(campaign.Name, trigger);
            var keyCampaign = CounterKey.GetKey(campaignName: campaign.Name);
            using (_counters.LtoStart.MultipleChangeScope())
            {
                _counters.LtoStart.IncrementAll(keyCampaign);
                _counters.LtoStart.IncrementAll(keyCampaignPerTrigger);
            }

            _offersPrefs.SetLastOfferStartTime(campaign.Name, currentTime);

            // Update trigger period activation time if empty
            var prefsKey = keyCampaignPerTrigger.ToString();
            if (_offersPrefs.GetLastOfferStartTime(prefsKey) == default)
            {
                _offersPrefs.SetLastOfferStartTime(prefsKey, currentTime);
            }

            AddActiveOffer(offer);

            if (conflictedOffer != null)
            {
                CompleteFinishedOffer(conflictedOffer);
            }

            SaveActiveOffers();
        }

        private bool ShouldBeCompleted(LtoModel offer, out string reason)
        {
            if (offer.Record == null)
            {
                reason = "Campaign not found";
                return true;
            }

            if (offer.EndTime < DateTime.UtcNow)
            {
                reason = "End time";
                return true;
            }
            if (!offer.Info.SubscriptionStatus.IsCorresponded(_generalPrefs.SubscriptionStatus.Value))
            {
                reason = "Subscription status";
                return true;
            }
            if (!offer.Info.InAppStatus.IsCorresponded(_generalPrefs.InAppStatus.Value))
            {
                reason = "InApp status";
                return true;
            }
            if (!offer.Info.AuthorizationStatus.IsCorresponded(_generalPrefs.AuthorizationStatus.Value))
            {
                reason = "Authorization status";
                return true;
            }
            if (!offer.Info.PayingStatus.IsCorresponded(_generalPrefs.SubscriptionStatus.Value, _generalPrefs.InAppStatus.Value))
            {
                reason = "Paying status";
                return true;
            }
            if (!offer.Info.PurchaseLimits.IsCorresponded(_generalPrefs.AllUsedProducts))
            {
                reason = "Purchase limits";
                return true;
            }
            if (!offer.Campaign.IsCampaignRelevant(_platform))
            {
                reason = "Campaign relevance";
                return true;
            }

            if (!offer.Campaign.IsWithinAppVersionLimits(_appVersionProvider.AppVersion))
            {
                reason = "App version limits";
                return true;
            }

            reason = "";
            return false;
        }

        private void SaveActiveOffers()
        {
            _ltoStorage.SaveActiveOffers(_activeOffers.Offers);
        }

        private void AddActiveOffer(LtoModel offer)
        {
            var delay = offer.EndTime - DateTime.UtcNow;
            if (delay <= TimeSpan.Zero)
            {
                _logger.LogWarning($"[LTO] Can't schedule offer completion, since end time is up: {offer.CampaignName}");
                return;
            }

            _activeOffers.Add(offer);
            OnAdded?.Invoke(offer);
            ScheduleOfferCompletion(offer, delay);
        }

        private void UpdateActiveOffer(LtoModel offer, CampaignRecord record)
        {
            var delay = offer.EndTime - DateTime.UtcNow;
            OnUpdated?.Invoke(offer);
            ScheduleOfferCompletion(offer, delay);
        }

        private void RemoveActiveOffer(LtoModel offer)
        {
            if (offer == null)
            {
                return;
            }
            _activeOffers.Remove(offer);
            OnRemoved?.Invoke(offer);
        }

        private void CompleteFinishedOffer(LtoModel offer)
        {
            _logger.Log($"[LTO] {nameof(CompleteFinishedOffer)} offer = {offer.CampaignName}");

            _campaignsTracker.ResetActivationCounters(offer);

            var endTime = TimeExtensions.MinOf(DateTime.UtcNow, offer.EndTime);
            _offersPrefs.SetLastOfferEndTime(offer.CampaignName, endTime);
            OnFinished?.Invoke(offer);
        }

        private void ScheduleOfferCompletion([NotNull] LtoModel offer, TimeSpan delay)
        {
            var cancellationTokenSource = new CancellationTokenSource();
            _activeOffers.AttachCancellationTokenSource(offer.CampaignName, cancellationTokenSource);
            UniTask.Delay(delay, delayType: DelayType.Realtime, cancellationToken: cancellationTokenSource.Token).ContinueWith(() =>
            {
                RemoveActiveOffer(offer);
                CompleteFinishedOffer(offer);
                SaveActiveOffers();
            });
        }
    }
}