using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using Magify.Model;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using UnityEngine;

namespace Magify
{
    internal static class EventParamExtensions
    {
        internal static bool Match(this EventParam param, IReadOnlyDictionary<string, object> eventParams)
        {
            if (!eventParams.TryGetValue(param.Name, out var rawValue))
                return false;

            return param.Type switch
            {
                "bool" => param.MatchBool(rawValue),
                "int" => param.MatchInt(rawValue),
                "float" => param.MatchFloat(rawValue),
                "string" => param.MatchString(rawValue),
                "choice" => param.MatchChoice(rawValue),
                "date" => param.MatchDate(rawValue),
                "datetime" => param.MatchDateTime(rawValue),
                "array" => param.MatchArray(rawValue),
                "int_counter" => param.MatchIntCounter(rawValue),
                _ => false
            };
        }

        private static bool MatchBool(this EventParam param, object rawValue)
        {
            if (rawValue is bool inputValue && param.Value is bool paramValue)
            {
                return param.Operation switch
                {
                    "is" => inputValue == paramValue,
                    "isnot" => inputValue != paramValue,
                    _ => false
                };
            }
            return false;
        }

        private static bool MatchInt(this EventParam param, object rawValue)
        {
            if (!NumberExtensions.TryToInt32(rawValue, out var inputValue) || !NumberExtensions.TryToInt32(param.Value, out var paramValue))
                return false;

            return param.Operation switch
            {
                "=" => inputValue == paramValue,
                "!=" => inputValue != paramValue,
                "<" => inputValue < paramValue,
                "<=" => inputValue <= paramValue,
                ">" => inputValue > paramValue,
                ">=" => inputValue >= paramValue,
                "//" => MatchPosition(inputValue, paramValue, param.ValueFrom, param.ValueTo),
                _ => false
            };
        }

        private static bool MatchPosition(int value, int period, int? from, int? to)
        {
            if (to != null && value > to)
                return false;

            var start = from ?? period;
            return value switch
            {
                _ when value == start => true,
                _ when value > start => period != 0 && (value - start) % period == 0,
                _ => false
            };
        }

        private static bool MatchFloat(this EventParam param, object rawValue)
        {
            if (!NumberExtensions.TryToSingle(rawValue, out var inputValue) || !NumberExtensions.TryToSingle(param.Value, out var paramValue))
                return false;

            return param.Operation switch
            {
                "=" => inputValue == paramValue,
                "!=" => inputValue != paramValue,
                "<" => inputValue < paramValue,
                "<=" => inputValue <= paramValue,
                ">" => inputValue > paramValue,
                ">=" => inputValue >= paramValue,
                _ => false
            };
        }

        private static bool MatchString(this EventParam param, object rawValue)
        {
            if (rawValue is not string inputValue || param.Value is not string paramValue)
                return false;

            return param.Operation switch
            {
                "=" => inputValue.Equals(paramValue),
                "!=" => !inputValue.Equals(paramValue),
                "match" => System.Text.RegularExpressions.Regex.IsMatch(inputValue, paramValue),
                _ => false
            };
        }

        internal static bool MatchChoice(this EventParam param, object rawValue)
        {
            if (rawValue is not string inputValue || !TryParseChoice(param.Value, out var paramValue))
                return false;

            return param.Operation switch
            {
                "in" => paramValue.Contains(inputValue),
                "notin" => !paramValue.Contains(inputValue),
                _ => false
            };
        }

        private static bool TryParseChoice(object value, out IEnumerable<string> result)
        {
            if (value is IEnumerable<string> enumerable)
            {
                result = enumerable;
                return true;
            }
            if (value is JArray jArray)
            {
                result = jArray.Values<string>();
                return true;
            }

            MagifyLogger.Get().LogError($"Unexpected choice value: {value} (type: {value?.GetType()}, json: {JsonFacade.SerializeObject(value, Formatting.Indented)})");
            result = default;
            return false;
        }

        private static bool MatchDate(this EventParam param, object rawValue)
        {
            if (param.Value is not string value || !DateTime.TryParseExact(value, "yyyy-MM-dd", null, DateTimeStyles.None, out var paramValue))
                return false;

            if (rawValue is not string input || !DateTime.TryParseExact(input, "yyyy-MM-dd", null, DateTimeStyles.None, out var inputValue))
                return false;

            return param.Operation switch
            {
                "=" => inputValue == paramValue,
                "!=" => inputValue != paramValue,
                "<" => inputValue < paramValue,
                "<=" => inputValue <= paramValue,
                ">" => inputValue > paramValue,
                ">=" => inputValue >= paramValue,
                _ => false
            };
        }

        private static bool MatchDateTime(this EventParam param, object rawValue)
        {
            if (!NumberExtensions.TryToInt64(rawValue, out var inputValue) || !NumberExtensions.TryToInt64(param.Value, out var paramValue))
                return false;

            return param.Operation switch
            {
                "=" => inputValue == paramValue,
                "!=" => inputValue != paramValue,
                "<" => inputValue < paramValue,
                "<=" => inputValue <= paramValue,
                ">" => inputValue > paramValue,
                ">=" => inputValue >= paramValue,
                _ => false
            };
        }

        private static bool MatchArray(this EventParam param, object rawValue)
        {
            if (!NumberExtensions.TryToInt32(rawValue, out var inputValue) || param.Value is not string paramValue)
                return false;

            var paramValues = paramValue.Split(',')
                .Select(part => int.TryParse(part, out var num) ? (int?)num : null)
                .Where(num => num.HasValue)
                .Select(num => num.Value)
                .ToList();

            return param.Operation switch
            {
                "in" => paramValues.Contains(inputValue),
                "notin" => !paramValues.Contains(inputValue),
                _ => false
            };
        }

        private static bool MatchIntCounter(this EventParam param, object rawValue)
        {
            return param.Operation switch
            {
                "in" or "notin" => param.MatchArray(rawValue),
                "=" or "!=" or "<" or "<=" or ">" or ">=" or "//" => param.MatchInt(rawValue),
                _ => false
            };
        }
    }
}