using JetBrains.Annotations;
using Magify.Model;

namespace Magify
{
    internal abstract class ProductParser
    {
        public virtual bool IsValidModel([NotNull] NestedCampaign nestedCampaign)
        {
            return !string.IsNullOrWhiteSpace(nestedCampaign.ProductId);
        }

        public abstract ProductDef ParseProduct([NotNull] NestedCampaign nestedCampaign, [NotNull] string clientId);
    }

    internal abstract class ProductParser<T> : ProductParser
        where T : ProductDef
    {
        public sealed override ProductDef ParseProduct(NestedCampaign nestedCampaign, string clientId)
        {
            return Parse(nestedCampaign, clientId);
        }

        protected abstract T Parse(NestedCampaign nestedCampaign, string clientId);
    }

    #region Realizations

    internal class RewardProductParser : ProductParser<RewardProduct>
    {
        protected override RewardProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new RewardProduct(nestedCampaign.ProductId!, nestedCampaign.Count ?? 0, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class BonusProductParser : ProductParser<BonusProduct>
    {
        protected override BonusProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new BonusProduct(nestedCampaign.ProductId!, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class InAppProductParser : ProductParser<InAppProduct>
    {
        protected override InAppProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new InAppProduct(nestedCampaign.ProductId!, nestedCampaign.Type == ProductType.Consumable, nestedCampaign.StoreName, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class SubscriptionProductParser : ProductParser<SubscriptionProduct>
    {
        protected override SubscriptionProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new SubscriptionProduct(nestedCampaign.ProductId!, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class InfoProductParser : ProductParser<InfoProduct>
    {
        protected override InfoProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new InfoProduct(nestedCampaign.ProductId!, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class CrossPromoProductParser : ProductParser<CrossPromoProduct>
    {
        public override bool IsValidModel(NestedCampaign nestedCampaign)
        {
            return base.IsValidModel(nestedCampaign)
                && !string.IsNullOrWhiteSpace(nestedCampaign.PromotedApplication?.BundleId)
                && !string.IsNullOrWhiteSpace(nestedCampaign.PromotedApplication?.RawPromoLink);
        }

        protected override CrossPromoProduct Parse(NestedCampaign nestedCampaign, string clientId)
        {
            var promoLink = nestedCampaign.PromotedApplication.GetPromoLink(clientId);
            return new CrossPromoProduct(nestedCampaign.ProductId!, nestedCampaign.PromotedApplication!.BundleId!, promoLink, nestedCampaign.PromotedApplication?.Schemas, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class ExternalLinkProductParser : ProductParser<ExternalLinkProduct>
    {
        public override bool IsValidModel(NestedCampaign nestedCampaign)
        {
            return base.IsValidModel(nestedCampaign) && !string.IsNullOrWhiteSpace(nestedCampaign.LinkUrl);
        }

        protected override ExternalLinkProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new ExternalLinkProduct(nestedCampaign.ProductId!, nestedCampaign.LinkUrl!, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    internal class InternalLinkProductParser : ProductParser<InternalLinkProduct>
    {
        public override bool IsValidModel(NestedCampaign nestedCampaign)
        {
            return base.IsValidModel(nestedCampaign) && !string.IsNullOrWhiteSpace(nestedCampaign.LinkUrl);
        }

        protected override InternalLinkProduct Parse(NestedCampaign nestedCampaign, string _)
        {
            return new InternalLinkProduct(nestedCampaign.ProductId!, nestedCampaign.LinkUrl!, nestedCampaign.ExtractProductCreative(), nestedCampaign.ProductContext);
        }
    }

    #endregion
}