using System;
using JetBrains.Annotations;

namespace Magify
{
    internal class Counters
    {
        private readonly ICounter[] _counters;
        private readonly CountersStorage _storage;

        [CanBeNull]
        private readonly Func<CounterType, CounterScope, CounterKey, int?> _alternativeCountersGetter;

        public Counter<BonusesCounterScope> Bonuses { get; }
        public Counter<ClicksCounterScope> Clicks { get; }
        public Counter<EventsCounterScope> Events { get; }
        public Counter<ImpressionsCounterScope> Impressions { get; }
        public Counter<RewardsCounterScope> Rewards { get; }
        public Counter<NestedCounterScope> Nested { get; }
        public Counter<LtoStartCounterScope> LtoStart { get; }

        public Counters(CountersStorage storage, [CanBeNull] Func<CounterType, CounterScope, CounterKey, int?> alternativeCountersGetter = null)
        {
            _alternativeCountersGetter = alternativeCountersGetter;
            _storage = storage;
            _counters = new ICounter[Enum.GetValues(typeof(CounterType)).Length];
            Bonuses = CreateCounter<BonusesCounterScope>(CounterType.Bonuses);
            Clicks = CreateCounter<ClicksCounterScope>(CounterType.Clicks);
            Events = CreateCounter<EventsCounterScope>(CounterType.Events);
            Impressions = CreateCounter<ImpressionsCounterScope>(CounterType.Impressions);
            Rewards = CreateCounter<RewardsCounterScope>(CounterType.Rewards);
            Nested = CreateCounter<NestedCounterScope>(CounterType.Nested);
            LtoStart = CreateCounter<LtoStartCounterScope>(CounterType.LtoStart);
        }

        public ICounter this[CounterType type] => _counters[(int)type];

        private Counter<T> CreateCounter<T>(CounterType type) where T : unmanaged, Enum
        {
            var counter = new Counter<T>(_storage[type], _alternativeCountersGetter != null ? (scope, key) => _alternativeCountersGetter(type, scope, key) : null);
            _counters[(int)type] = counter;
            return counter;
        }

        public void ResetAll(CounterScope scope, Func<CounterKey, bool> predicate)
        {
            foreach (var counter in _counters)
            {
                counter?.ResetAll((s, k) => s == scope && predicate(k));
            }
        }

        public void ResetAll(CounterScope scope)
        {
            foreach (var counter in _counters)
            {
                counter?.ResetAll(scope);
            }
        }
    }
}