using System.IO;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class ContextFileStorage : FileStorage, ILegacyMigrator
    {
        [NotNull]
        private const string ConfigJson = "config.json";
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        [NotNull]
        private readonly object _fileLock = new();

        public ContextFileStorage(string storagePath) : base(storagePath)
        {
        }

        public ContextFileStorage(string folderName, string storagePath) : base(storagePath, folderName)
        {
        }

        void ILegacyMigrator.Migrate(MigrationData data)
        {
            _logger.Log($"Try to migrate context file from old path: {data.ContextPath}");
            if (Exists(ConfigJson))
            {
                _logger.Log($"Saved context file already exists at path {ConfigJson}. Skip migration.");
                return;
            }
            if (data.ContextPath == null || !File.Exists(data.ContextPath))
            {
                _logger.Log($"Old context file doesn't exists at path {data.ContextPath}. Skip migration.");
                return;
            }
            _logger.Log($"Old context file found at path {data.ContextPath}. Start migration to new file {ConfigJson}");
            lock (_fileLock)
            {
                CopyTextFrom(data.ContextPath, ConfigJson);
            }
            _logger.Log($"Old context file successfully migrated to new path {ConfigJson}.");
        }

        public void SaveContext(string json)
        {
            lock (_fileLock)
            {
                Save(ConfigJson, json);
            }
        }

        public string LoadContext()
        {
            lock (_fileLock)
            {
                return Load(ConfigJson);
            }
        }

        public void ResetContext()
        {
            lock (_fileLock)
            {
                Clear(ConfigJson);
            }
        }
    }
}