﻿using System;
using System.IO;
using System.IO.Compression;
using System.Text;
using JetBrains.Annotations;

namespace Magify
{
    internal class ToJsonToZipToBase64MagifyEncoder : IMagifyEncoder
    {
        private const string ZipEntryName = "AppState";

        [NotNull]
        public string Encode<T>([NotNull] T value, [NotNull] string encodingModePrefix)
            where T: class
        {
            // To json
            var json = JsonFacade.SerializeObject(value);

            // To zip
            var stateBytes = Encoding.UTF8.GetBytes(json); // prepare binary data for zipping
            using var memoryStream = new MemoryStream();
            using (var archive = new ZipArchive(memoryStream, ZipArchiveMode.Create, true))
            {
                using var zipStream = archive.CreateEntry(ZipEntryName, CompressionLevel.Optimal)!.Open();
                zipStream.Write(stateBytes, 0, stateBytes.Length);
            }

            // To base64
            var zipStateBytes = memoryStream.ToArray();
            // Calculate future base64 string size to use Convert.ToBase64CharArray() and avoid re-allocations
            var zipStateBytesLength = Base64Utils.CalculateAndValidateOutputLength(zipStateBytes.Length, false);
            var prefixLength = encodingModePrefix.Length;

            var result = new char[prefixLength + zipStateBytesLength];
            encodingModePrefix.CopyTo(0, result, 0, prefixLength); // Write prefix

            Convert.ToBase64CharArray(
                inArray: zipStateBytes, offsetIn: 0, zipStateBytes.Length, // Write all stateBytes...
                outArray: result, offsetOut: prefixLength); // with offset for the prefix chars
            return new string(result);
        }

        [CanBeNull]
        public T Decode<T>([CanBeNull] string encoded, int encodingModePrefixLength)
            where T: class
        {
            if (string.IsNullOrEmpty(encoded))
            {
                return null;
            }

            // From base64
            var span = (ReadOnlySpan<char>)encoded;
            var zipBytes = new byte[span.Length - encodingModePrefixLength];
            Convert.TryFromBase64Chars(span[encodingModePrefixLength..], zipBytes, out _); // Ignore prefix and read app state bytes

            // From zip
            using var memoryStream = new MemoryStream(zipBytes);
            using var archive = new ZipArchive(memoryStream, ZipArchiveMode.Read);
            var zipEntry = archive.Entries[0];
            using var zipStream = zipEntry.Open();
            using var reader = new StreamReader(zipStream, Encoding.UTF8);
            var json = reader.ReadToEnd();

            // From json
            return JsonFacade.DeserializeObject<T>(json);
        }
    }
}