using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using Magify.Types;
using UnityEngine;
using UnityEngine.Pool;
using EventType = Magify.Model.EventType;

namespace Magify
{
    internal class AnalyticsTracker : IAnalyticsTracker, ILegacyMigrator, IInitializable, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Analytics);

        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly IDictionary<EventType, IAnalyticsEventHandler> _eventHandlers = new Dictionary<EventType, IAnalyticsEventHandler>();
        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private AppVersionProvider _appVersionProvider;
        [NotNull]
        private readonly Subject<EventType> _onEventSent = new();
        [NotNull]
        private readonly Subject<(EventType, IAnalyticsEvent)> _onEventSentDetailed = new();

        public IObservable<EventType> OnEventSent => _onEventSent;
        public IObservable<(EventType Type, IAnalyticsEvent Event)> OnEventSentDetailed => _onEventSentDetailed;
        [NotNull]
        public LevelStateInfo LevelState { get; } = LevelStateInfo.Empty;

        public AnalyticsTracker(
            [NotNull] IServerApi serverApi,
            [NotNull] GeneralPrefs generalPrefs,
            [NotNull] PlatformAPI platform,
            [NotNull] AppStatePrefs appStatePrefs,
            string storagePath,
            [NotNull] AppVersionProvider appVersion)
        {
            _serverApi = serverApi;
            _generalPrefs = generalPrefs;
            _appStatePrefs = appStatePrefs;
            _platform = platform;
            _appVersionProvider = appVersion;
            CreateEventHandlers(storagePath);
        }

        void ILegacyMigrator.Migrate([NotNull] MigrationData data)
        {
            migrate(EventType.AppLaunch, data.AppLaunchEvents);
            migrate(EventType.SessionEvent, data.CustomSessionEvents);
            migrate(EventType.AppBackgroundEvent, data.AppBackgroundEvents);
            migrate(EventType.Transaction, data.TransactionEvents);
            migrate(EventType.FirebaseMapping, data.FirebaseMappingEvents);
            migrate(EventType.MailingStatusMapping, data.MailingStatusMappingEvents);
            migrate(EventType.Impression, data.ImpressionEvents);
            migrate(EventType.ApplovinAdsImpression, data.ApplovinAdsImpressionEvents);
            migrate(EventType.IronSourceAdsImpression, data.IronSourceAdsImpressionEvents);
            migrate(EventType.ImpressionFail, data.ImpressionFailEvents);
            migrate(EventType.TrialActivation, data.TrialActivationEvents);
            migrate(EventType.PaidSubscriptionActivation, data.PaidSubscriptionActivationEvents);
            migrate(EventType.InApp, data.InAppEvents);
            migrate(EventType.Click, data.ClickEvents);
            migrate(EventType.AdsClick, data.AdsClickEvents);

            void migrate<T>(EventType eventType, [CanBeNull] IReadOnlyCollection<T> events) where T : class, IAnalyticsEvent
            {
                _logger.Log($"Trying to migrate {eventType} events");
                if (events == null)
                {
                    _logger.Log($"No {eventType} events to migrate");
                    return;
                }

                var eventHandler = (IAnalyticsEventHandler<T>)_eventHandlers[eventType]!;
                if (eventHandler.Storage.Exists())
                {
                    _logger.Log($"Storage for {eventType} events already exists. Skip migration");
                    return;
                }
                eventHandler.Storage.SaveEvents(events);
                _logger.Log($"Successfully migrated {events.Count} {eventType} events:\n{JsonFacade.SerializeObject(events)}");
            }
        }

        void IInitializable.Initialize()
        {
            _generalPrefs.FirebaseInstanceId
                .SkipLatestValueOnSubscribe()
                .Subscribe(TrackFirebaseMappingEvent)
                .AddTo(_disposables);

            foreach (var (_, value) in _eventHandlers)
            {
                value!.Initialize();
            }
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();
            _eventHandlers.Values.OfType<IDisposable>().ForEach(d => d?.Dispose());
        }

        public void HandeWentForeground()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value!.OnForeground();
            }
        }

        public void HandeWentBackground()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value!.OnBackground();
            }
        }

        public void ForceSync()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value!.ForceSync();
            }
        }

        public void SetupAnalyticsConfig(AnalyticsConfiguration config)
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value!.SetupAnalyticsConfig(config);
            }
        }

        public void ResetAnalyticsConfig()
        {
            foreach (var (_, value) in _eventHandlers)
            {
                value!.ResetAnalyticsConfig();
            }
        }

        #region Track Events

        public void TrackCustomSessionEvent(string eventName, [CanBeNull] Dictionary<string, object> parameters = null)
        {
            var parametersDictionary = parameters != null ? DictionaryPool<string, object>.Get() : null;
            if (parametersDictionary != null)
            {
                parametersDictionary!.Copy(parameters);
            }

            var customSessionEvent = new CustomSessionEvent(
                eventName,
                _generalPrefs.SessionId.Value,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                parametersDictionary,
                LevelState,
                _generalPrefs.GlobalSessionCounter.Value,
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
            HandleEvent(EventType.SessionEvent, customSessionEvent);
        }

        public void TrackTransaction(Transaction transaction, IEnumerable<BonusInfo> bonuses, ProductInfo product = null, CampaignImpression impression = null)
        {
            var events = bonuses.Select(bonus => new TransactionEvent(
                transaction,
                bonus,
                _generalPrefs.GlobalSessionCounter.Value,
                impression,
                product,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                PackageInfo.Version
            ));
            HandleEvents(EventType.Transaction, new LinkedList<TransactionEvent>(events));
        }

        public void TrackFirebaseMappingEvent(string appInstanceId)
        {
            if (appInstanceId == null) return;
            var @event = new FirebaseMappingEvent
            {
                AppInstanceId = appInstanceId,
                SdkVersion = PackageInfo.Version,
            };
            HandleEvent(EventType.FirebaseMapping, @event);
        }

        public void TrackMailingStatusMappingEvent(string email)
        {
            var @event = new MailingStatusMappingEvent
            {
                Email = email,
                AppVersion = _appVersionProvider.AppVersion,
                SdkVersion = PackageInfo.Version,
            };
            HandleEvent(EventType.MailingStatusMapping, @event);
        }

        public void TrackApplovinAdsImpression(CampaignImpression impression, ApplovinAdsImpression data)
        {
            var @event = new AdsImpressionEvent(
                impression,
                data,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
            HandleEvent(EventType.ApplovinAdsImpression, @event);
        }

        public void TrackIronSourceAdsImpression(CampaignImpression impression, IronSourceAdsImpression data)
        {
            var @event = new AdsImpressionEvent(
                impression,
                data,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
            HandleEvent(EventType.IronSourceAdsImpression, @event);
        }

        public void TrackPurchaseEvent(EventType type, CampaignImpression impression, PurchaseInfo purchase)
        {
            var analyticsEvent = new ProductPurchaseEvent(
                impression,
                purchase,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _generalPrefs.SessionId.Value,
                purchase.CustomStoreFront is { IsSome: true } ? purchase.CustomStoreFront.Value : _platform.GetStoreCountry(),
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
            HandleEvent(type, analyticsEvent);
        }

        public void TrackImpressionFailEvent(CampaignImpression impression, string reason)
        {
            var @event = new ImpressionFailEvent(
                impression,
                reason,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
            HandleEvent(EventType.ImpressionFail, @event);
        }

        public void TrackAppLaunch()
        {
            var @event = new AppLaunchEvent
            {
                SubscriptionStatus = _generalPrefs.SubscriptionStatus.Value,
                InAppStatus = _generalPrefs.InAppStatus.Value,
                AppVersion = _appVersionProvider.AppVersion,
            };
            HandleEvent(EventType.AppLaunch, @event);
        }

        public void TrackAppBackgroundEvent(long openTime, long closeTime, string appVersion)
        {
            var @event = new AppBackgroundEvent
            {
                SessionId = _generalPrefs.SessionId.Value,
                OpenTimestamp = openTime.TimestampFromLongMills(),
                CloseTimestamp = closeTime.TimestampFromLongMills(),
                AppVersion = appVersion,
                SdkVersion = PackageInfo.Version,
            };
            HandleEvent(EventType.AppBackgroundEvent, @event);
        }

        public void TrackUsedApplicationDefaultFeaturesEvent([NotNull] string featureName)
        {
            var @event = new UsedApplicationDefaultFeaturesEvent
            {
                FeatureName = featureName,
                SdkVersion = PackageInfo.Version,
            };
            HandleEvent(EventType.UsedApplicationDefaultFeatures, @event);
        }

        public void TrackCommonEvents(EventType type, [NotNull] IReadOnlyList<CampaignImpression> impressions)
        {
            var models = new List<CampaignImpressionEvent>(impressions.Count);
            models.AddRange(impressions.Select(CreateCampaignImpressionEvent));
            HandleEvents(type, models);
        }

        public void TrackCommonEvent(EventType type, CampaignImpression impression)
        {
            var model = CreateCampaignImpressionEvent(impression);
            HandleEvent(type, model);
        }

        private CampaignImpressionEvent CreateCampaignImpressionEvent(CampaignImpression impression)
        {
            return new CampaignImpressionEvent(
                impression,
                _generalPrefs.SubscriptionStatus.Value,
                _generalPrefs.InAppStatus.Value,
                LevelState,
                _generalPrefs.SessionId.Value,
                _appVersionProvider.AppVersion,
                PackageInfo.Version);
        }

        #endregion

        private void HandleEvent<T>(EventType eventType, T @event)
            where T : class, IAnalyticsEvent
        {
            ((IAnalyticsEventHandler<T>)_eventHandlers[eventType])!.HandleEvent(@event);
            _onEventSent.OnNext(eventType);
            _onEventSentDetailed.OnNext((eventType, @event));
        }

        private void HandleEvents<T>(EventType eventType, IReadOnlyCollection<T> events)
            where T : class, IAnalyticsEvent
        {
            ((IAnalyticsEventHandler<T>)_eventHandlers[eventType])!.HandleEvents(events);
            _onEventSent.OnNext(eventType);
            events.ForEach(@event => _onEventSentDetailed.OnNext((eventType, @event)));
        }

        private void CreateEventHandlers(string storagePath)
        {
            foreach (var eventTypeFlag in Enum.GetValues(typeof(EventType)).Cast<EventType>())
            {
                _eventHandlers.Add(eventTypeFlag, CreateEventHandleByEventType(eventTypeFlag, storagePath));
            }
        }

        private IAnalyticsEventHandler CreateEventHandleByEventType(EventType eventType, string storagePath)
        {
            return eventType switch
            {
                EventType.AppLaunch => create<AppLaunchEvent>(),
                EventType.SessionEvent => create<CustomSessionEvent>(),
                EventType.AppBackgroundEvent => create<AppBackgroundEvent>(),
                EventType.Transaction => create<TransactionEvent>(),
                EventType.FirebaseMapping => create<FirebaseMappingEvent>(),
                EventType.MailingStatusMapping => create<MailingStatusMappingEvent>(),
                EventType.ImpressionFail => create<ImpressionFailEvent>(),
                EventType.Impression => create<CampaignImpressionEvent>(),
                EventType.Click => create<CampaignImpressionEvent>(),
                EventType.AdsClick => create<CampaignImpressionEvent>(),
                EventType.ApplovinAdsImpression => create<AdsImpressionEvent>(),
                EventType.IronSourceAdsImpression => create<AdsImpressionEvent>(),
                EventType.InApp => create<ProductPurchaseEvent>(),
                EventType.TrialActivation => create<ProductPurchaseEvent>(),
                EventType.PaidSubscriptionActivation => create<ProductPurchaseEvent>(),
                EventType.UsedApplicationDefaultFeatures => create<UsedApplicationDefaultFeaturesEvent>(),
                _ => throw new ArgumentOutOfRangeException(nameof(eventType), eventType, null),
            };

            [NotNull]
            EventHandler<T> create<T>()
                where T : class, IAnalyticsEvent
            {
                return new EventHandler<T>(eventType, _serverApi, _appStatePrefs, storagePath);
            }
        }
    }
}