﻿using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Networking;

namespace Magify
{
    internal class RemoteStorageNetworkClient : IRemoteStorageNetworkClient
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Storage);

        public async UniTask<StorageResultCode> SendWebRequest([NotNull] UnityWebRequest request, CancellationToken cancellationToken)
        {
            try
            {
                await request.SendWebRequest().WithCancellation(cancellationToken).AsUniTask();
            }
            catch (UnityWebRequestException exception)
            {
                _logger.Log($"{nameof(SendWebRequest)} finished by unity exception: {exception.Error}\n{request.url}");
            }

            _logger.Log($"{nameof(SendWebRequest)} finished with result {request.result}\n{request.url}");
            if (request.result != UnityWebRequest.Result.Success)
            {
                _logger.LogError($"Can't download content. Error {request.error}\n{request.url}");
                var errorCore = request.result switch
                {
                    UnityWebRequest.Result.ConnectionError => StorageResultCode.ConnectionError,
                    UnityWebRequest.Result.ProtocolError => StorageResultCode.ProtocolError,
                    UnityWebRequest.Result.DataProcessingError => StorageResultCode.DataProcessingError,
                    _ => StorageResultCode.UnknownError,
                };
                return errorCore;
            }
            return StorageResultCode.Success;
        }
    }

    public interface IRemoteStorageNetworkClient
    {
        UniTask<StorageResultCode> SendWebRequest([NotNull] UnityWebRequest request, CancellationToken cancellationToken);
    }
}