﻿using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Synchronization
        {
            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Allows you to handle successful state synchronization.
            /// </summary>
            /// <returns>
            /// <see cref="SyncStateResult"/>, that indicates restore result <br/>
            /// <see cref="SyncStateResult"/>.<see cref="SyncStateResult.Kind"/> might be only
            /// <see cref="SyncStateResult.ResultKind.Success"/> or <see cref="SyncStateResult.ResultKind.NothingToRestore"/> here.
            /// </returns>
            /// <remarks>
            /// If you want to have more control over the result of the restore, use the return value <see cref="RestoreState"/>.
            /// </remarks>
            [NotNull]
            public static IObservable<SyncStateResult> OnRestoreStateCompleted
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(OnRestoreStateCompleted));
                    return MagifyPlatformAPI!.OnRestoreStateCompleted;
                }
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Provides automatic restore state info.
            /// </summary>
            /// <returns>
            /// <see cref="AutoRestoreStateInfo"/>, that indicates restore result <br/>
            /// <see cref="AutoRestoreStateInfo"/>.<see cref="AutoRestoreStateInfo.StateKind"/>: <br/>
            /// <ul>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.NotStarted"/> <b>(default)</b> if auto restore has not been started within the current application session</li>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.InProgress"/> if auto restore is in progress (including retries)</li>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.Success"/> if auto restore is successfully completed</li>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.NothingToRestore"/> if auto restore is successfully finished, but there is no data to restore</li>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.Failed"/> if auto restore is failed, see details inside</li>
            ///     <li><see cref="AutoRestoreStateInfo.StateKind.Cancelled"/> if auto restore is cancelled</li>
            /// </ul>
            /// </returns>
            /// <remarks>
            /// Doesn't save between restarts.
            /// </remarks>
            [NotNull]
            public static IReadOnlyReactiveProperty<AutoRestoreStateInfo> AutoRestoreStateInfo
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(AutoRestoreStateInfo));
                    return MagifyPlatformAPI!.AutoRestoreStateInfo;
                }
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Allows you to control automatic restore state.
            /// </summary>
            /// <remarks>
            /// Enabled by default. <br/>
            /// Will be saved between restarts. <br/>
            /// <see cref="AutoRestoreStateInfo"/> will be <see cref="AutoRestoreStateInfo.StateKind.Cancelled"/> by setting this value to `false` while auto-restore is in progress.
            /// </remarks>
            public static bool IsAutoRestoreStateEnabled
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(IsAutoRestoreStateEnabled));
                    return MagifyPlatformAPI!.IsAutoRestoreStateEnabled.Value;
                }
                set
                {
                    ThrowIfMagifyIsNotReady(nameof(IsAutoRestoreStateEnabled));
                    MagifyPlatformAPI!.IsAutoRestoreStateEnabled.Value = value;
                }
            }

            [NotNull]
            internal static IReadOnlyReactiveProperty<long> LastSuccessfulSyncProgressUnixTimeSeconds
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(LastSuccessfulSyncProgressUnixTimeSeconds));
                    return MagifyPlatformAPI!.LastSuccessfulSyncProgressUnixTimeSeconds;
                }
            }

            internal static bool ShouldSendLastSyncProgressTime
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(ShouldSendLastSyncProgressTime));
                    return MagifyPlatformAPI!.ShouldSendLastSyncProgressTime;
                }
                set
                {
                    ThrowIfMagifyIsNotReady(nameof(ShouldSendLastSyncProgressTime));
                    MagifyPlatformAPI!.ShouldSendLastSyncProgressTime = value;
                }
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Requests saving the state of the application. You shouldn't call more than one operation in one time.
            /// </summary>
            /// <param name="weight"> A weight that determines how far a player has gone in the game.
            ///     <ul>
            ///         <li> If the weight parameter is specified:
            ///             <ul>
            ///                 <li>if the value <b>is greater than</b> the value on the server - progress <b>will</b> be saved</li>
            ///                 <li>if the value <b>is less than</b> the value on the server - progress <b>won’t</b> be saved</li>
            ///             </ul>
            ///         </li>
            ///         <li> If the weight parameter is not specified:
            ///             <ul>
            ///                 <li>if <b>there is any</b> value on the server - progress <b>won’t</b> be saved</li>
            ///             </ul>
            ///         </li>
            ///         <li> In the both cases:
            ///             <ul>
            ///                 <li>if there is no value (null) on the server - progress will be saved</li>
            ///             </ul>
            ///         </li>
            ///     </ul>
            ///  <br/>
            /// </param>
            /// <param name="cancellationToken">Pass cancellation token if you want to cancel this operation, for example by timeout.</param>
            /// <returns>
            /// Result of the save operation.
            /// </returns>
            /// <remarks>
            /// Remember that you should pass <see cref="MagifyConfig.ClientStateConfig"/> with implemented <see cref="IAppStateProvider"/> to control which state will be saved. <br/>
            /// Magify will also save its state with your data.
            /// </remarks>
            // ReSharper disable once AnnotationRedundancyAtValueType
            // ReSharper disable once ContainerAnnotationRedundancy
            [NotNull, ItemNotNull]
            public static UniTask<SyncStateResult> SaveState([CanBeNull] int? weight, CancellationToken cancellationToken)
            {
                ThrowIfMagifyIsNotReady(nameof(SaveState));
                if (weight < 0)
                {
                    throw new MagifyArgumentOutOfRangeException($"{nameof(SaveState)}(int? weight)", ", it mustn't be less than zero");
                }
                _logger!.Log($"Save state with {nameof(weight)}={weight?.ToString() ?? "null"}");
                return MagifyPlatformAPI!.SaveState(weight, cancellationToken);
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Requests restoring a state of the application. You shouldn't call more than one operation in one time.
            /// </summary>
            /// <param name="weight">
            /// A weight that determines how far a player has gone in the game.
            ///     <ul>
            ///         <li> If the weight parameter is specified:
            ///             <ul>
            ///                 <li>if the value <b>is less than</b> the value on the server - progress <b>will</b> be loaded</li>
            ///                 <li>if the value <b>is greater than</b> the value on the server - progress <b>won’t</b> be loaded</li>
            ///             </ul>
            ///         </li>
            ///         <li> If the weight parameter <b>is not specified</b> - progress <b>will</b> be loaded </li>
            ///     </ul>
            /// </param>
            /// <param name="cancellationToken">Pass cancellation token if you want to cancel this operation, for example by timeout.</param>
            /// <returns>
            /// Result of the restore operation.
            /// </returns>
            /// <remarks>
            /// Remember that you should pass <see cref="MagifyConfig.ClientStateConfig"/> with implemented <see cref="IAppStateProvider"/> to handle restored state. <br/>
            /// Magify will also restore its state with your data (but only within one platform).
            /// </remarks>
            // ReSharper disable once AnnotationRedundancyAtValueType
            // ReSharper disable once ContainerAnnotationRedundancy
            [NotNull, ItemNotNull]
            public static UniTask<SyncStateResult> RestoreState([CanBeNull] int? weight, CancellationToken cancellationToken)
            {
                ThrowIfMagifyIsNotReady(nameof(RestoreState));
                if (weight < 0)
                {
                    throw new MagifyArgumentOutOfRangeException($"{nameof(RestoreState)}(int? weight)", ", it mustn't be less than zero");
                }
                _logger!.Log($"Restore state with {nameof(weight)}={weight?.ToString() ?? "null"}");
                return MagifyPlatformAPI!.RestoreState(weight, cancellationToken);
            }
        }
    }
}