using JetBrains.Annotations;

namespace Magify
{
    public class MagifyConfig
    {
        /// <summary>
        /// Magify application name for iOS
        /// </summary>
        [CanBeNull]
        public string AppNameIOS { get; init; }

        /// <summary>
        /// Path in StreamingAssets to default application config json for current version for iOS
        /// </summary>
        [CanBeNull]
        public string ConfigPathIOS { get; init; }

        /// <summary>
        /// Magify application name for Android
        /// </summary>
        [CanBeNull]
        public string AppNameGP { get; init; }

        /// <summary>
        /// Path in StreamingAssets to default application config json for current version for Android
        /// </summary>
        [CanBeNull]
        public string ConfigPathGP { get; init; }

        /// <summary>
        /// SDK data storage path
        /// </summary>
        [NotNull]
        internal string StoragePath { get; init; } = PackageInfo.PersistentPath;

        /// <summary>
        /// Configuration that will be used to save and restore client state
        /// </summary>
        [NotNull]
        public ClientStateConfig ClientStateConfig { get; init; } = new();

        /// <summary>
        /// Custom client id. Make sure to use same client id for all services (AppsFlyer, Adjust, Applovin and etc.)
        /// </summary>
        public CustomClientId CustomClientId { get; init; }

        /// <summary>
        /// Use it for emulating campaigns in editor
        /// </summary>
        public MagifyEditorMoq EditorMoq { get; init; }

        /// <summary>
        /// If you want to use the full functionality of the Magify SDK, use this flag
        /// </summary>
        public bool UseAdvancedVersion { get; init; } = false;

        /// <summary>
        /// If you want to use only moq use this flag
        /// </summary>
        public bool IsDeveloperMode { get; init; }

        /// <summary>
        /// Use it for emulating device in editor
        /// </summary>
        public EditorDevice EditorDevice { get; init; }

        /// <summary>
        /// Settings for Magify presenter canvas
        /// </summary>
        [CanBeNull]
        public IPresenterSettings PresenterSettings { get; init; }

        /// <summary>
        /// Default environment. You can change it later using <see cref="MagifyManager.Environment"/>
        /// </summary>
        public Environment Environment { get; init; } = Environment.Production;

        /// <summary>
        /// Indicates whether the current environment is running in Sandbox mode.
        /// </summary>
        public bool IsSandbox { get; init; }

        /// <inheritdoc cref="ConfigScope"/>
        /// <remarks>
        /// <inheritdoc cref="ConfigScope" path="/remarks"/> <br/>
        /// You can change it later using <see cref="MagifyManager"/>.<see cref="MagifyManager.RemoteConfigScopes"/>
        /// </remarks>
        public ConfigScope RemoteConfigScopes { get; init; }

        /// <summary>
        /// Enables or disables sync state (app progress sync)
        /// </summary>
        /// <remarks>If the value is null, default value(false) or saved value from previous session will be used.</remarks>
        [CanBeNull]
        public bool? SyncStateEnabled { get; init; }

        /// <summary>
        /// Enables or disables automatic state restoring
        /// </summary>
        public bool IsAutoRestoreStateEnabled { get; init; } = true;

        /// <summary>
        /// Allows to pass loading of ClientId from native cloud storage
        /// </summary>
        public bool SkipClientIdFromCloudLoading { get; init; } = false;

        /// <summary>
        /// Allows to set timeout for loading of ClientId from native cloud storage (in seconds)
        /// </summary>
        public uint ClientIdFromCloudLoadingTimeout { get; init; } = 30;

        /// <summary>
        /// Default SubscriptionStatus. You can change it later using <see cref="MagifyManager.SubscriptionStatus"/>
        /// </summary>
        public SubscriptionStatus SubscriptionStatus { get; init; } = SubscriptionStatus.Inactive;

        /// <summary>
        /// Default AuthorizationStatus. You can change it later using <see cref="MagifyManager.AuthorizationStatus"/>
        /// </summary>
        public AuthorizationStatus AuthorizationStatus { get; set; } = AuthorizationStatus.Unauthorized;

        /// <summary>
        /// State of user consent to GDPR
        /// </summary>
        /// <remarks>If the value is null, default value(false) or saved value from previous session will be used.</remarks>
        [CanBeNull]
        public bool? IsGdprApplied { get; set; }

        /// <summary>
        /// State of user consent to ATT
        /// </summary>
        /// <remarks>If the value is null, default value(false) or saved value from previous session will be used.</remarks>
        [CanBeNull]
        public bool? IsAttAuthorized { get; set; }

        /// <summary>
        /// Default logging status. You can change it later using <see cref="MagifyManager.IsLoggingEnabled"/>
        /// </summary>
        public bool IsLoggingEnabled { get; init; } = true;

        public string RevenuePerCountryPath { get; init; }
        public string RevenueLevelsPath { get; init; }
        public string DefaultCurrencyRatesPath { get; init; }
        public string SubscriptionMultipliersPath { get; init; }
    }

    public class MagifyDebugConfig
    {
        [CanBeNull]
        public string CustomAndroidId { get; init; }
        [CanBeNull]
        public string CustomBuildNumber { get; init; }
        [CanBeNull]
        public string CustomAppVersion { get; init; }
    }
}