﻿using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify.Rx
{
    [Serializable]
    public class ReactiveProperty<T> : IReactiveProperty<T>, IDisposable, IObserverLinkedList<T>
    {
        [NotNull]
        private static readonly IEqualityComparer<T> _defaultEqualityComparer = EqualityComparer<T>.Default;

        [SerializeField]
        private T value;

        [NonSerialized]
        private ObserverNode<T> _root;

        [NonSerialized]
        private ObserverNode<T> _last;

        [NonSerialized]
        private bool _isDisposed;

        [NotNull]
        protected IEqualityComparer<T> EqualityComparer => _defaultEqualityComparer;

        public T Value
        {
            get => value;
            set
            {
                if (!EqualityComparer.Equals(this.value, value))
                {
                    SetValue(value);
                    if (_isDisposed)
                    {
                        return;
                    }
                    RaiseOnNext(ref value);
                }
            }
        }

        // always true, allows empty constructor 'can' publish value on subscribe.
        // because sometimes value is deserialized from UnityEngine.
        public bool HasValue => true;

        public ReactiveProperty() : this(default)
        {
        }

        public ReactiveProperty(T initialValue)
        {
            SetValue(initialValue);
        }

        private void RaiseOnNext(ref T nextValue)
        {
            var node = _root;
            while (node != null)
            {
                node.OnNext(nextValue);
                node = node.Next;
            }
        }

        protected void SetValue(T newValue)
        {
            value = newValue;
        }

        public void SetValueAndForceNotify(T newValue)
        {
            SetValue(newValue);
            if (_isDisposed)
                return;

            RaiseOnNext(ref newValue);
        }

        public IDisposable Subscribe([NotNull] IObserver<T> observer)
        {
            if (_isDisposed)
            {
                observer.OnCompleted();
                return EmptyDisposable.Singleton;
            }

            // raise latest value on subscribe
            observer.OnNext(value);

            // subscribe node, node as subscription.
            var next = new ObserverNode<T>(this, observer);
            if (_root == null)
            {
                _root = _last = next;
            }
            else
            {
                _last!.Next = next;
                next.Previous = _last;
                _last = next;
            }
            return next;
        }

        void IObserverLinkedList<T>.UnsubscribeNode([NotNull] ObserverNode<T> node)
        {
            if (node == _root)
            {
                _root = node.Next;
            }
            if (node == _last)
            {
                _last = node.Previous;
            }

            if (node.Previous != null)
            {
                node.Previous.Next = node.Next;
            }
            if (node.Next != null)
            {
                node.Next.Previous = node.Previous;
            }
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (_isDisposed) return;

            var node = _root;
            _root = _last = null;
            _isDisposed = true;

            while (node != null)
            {
                node.OnCompleted();
                node = node.Next;
            }
        }

        public override string ToString()
        {
            return (value == null) ? "(null)" : value.ToString();
        }
    }
}