using System;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify.Rx
{
    internal sealed class MainThreadDispatcher : MonoBehaviour
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();
        [NotNull]
        private static readonly object _staticLock = new();
        [CanBeNull]
        private static MainThreadDispatcher _instance;
        private static bool _initialized;

        public static bool IsInitialized => _initialized && _instance != null;

        [NotNull]
        private static MainThreadDispatcher Instance
        {
            get
            {
                lock (_staticLock)
                {
                    return Initialize();
                }
            }
        }

        [NotNull]
        private readonly object _lock = new();

        [NotNull]
        public static MainThreadDispatcher Initialize()
        {
            var dispatcher = default(MainThreadDispatcher);

            if (!_initialized)
            {
                try
                {
                    dispatcher = FindObjectOfType<MainThreadDispatcher>();
                }
                catch
                {
                    // Throw exception when calling from a worker thread.
                    var ex = new Exception("Magify requires a MainThreadDispatcher component created on the main thread. Make sure it is added to the scene before calling Magify from a worker thread.");
                    _logger.LogException(ex);
                    throw ex;
                }

                if (dispatcher == null)
                {
                    new GameObject("MainThreadDispatcher").AddComponent<MainThreadDispatcher>();
                }
                else
                {
                    dispatcher.Awake();
                }
            }
            return _instance!;
        }

        private void Awake()
        {
            lock (_lock)
            {
                if (_instance == null)
                {
                    _instance = this;
                    _initialized = true;
                    DontDestroyOnLoad(gameObject);
                }
            }
        }

        private void OnDestroy()
        {
            lock (_lock)
            {
                if (_instance == this)
                {
                    _instance = FindObjectOfType<MainThreadDispatcher>();
                    _initialized = _instance != null;
                }
            }
        }

        private Subject<bool> _onApplicationFocus;

        private void OnApplicationFocus(bool focus)
        {
            if (_onApplicationFocus != null)
            {
                _onApplicationFocus.OnNext(focus);
            }
        }

        [NotNull]
        public static IObservable<bool> OnApplicationFocusAsObservable()
        {
            return Instance._onApplicationFocus ?? (Instance._onApplicationFocus = new Subject<bool>());
        }

        private Subject<bool> _onApplicationPause;

        private void OnApplicationPause(bool paused)
        {
            if (_onApplicationPause != null)
            {
                _onApplicationPause.OnNext(paused);
            }
        }

        [NotNull]
        public static IObservable<bool> OnApplicationPauseAsObservable()
        {
            return Instance._onApplicationPause ?? (Instance._onApplicationPause = new Subject<bool>());
        }
    }
}