#import <Foundation/Foundation.h>
#import <MagifyNative/MagifyNative.h>

#define GetNullableStringParam(_x_) ((_x_) != NULL ? [NSString stringWithUTF8String:_x_] : nil)

static char* cStringCopy(NSString* input)
{
    const char* string = [input UTF8String];
    return string ? strdup(string) : NULL;
}

@interface MagifyManager : NSObject
- (BOOL)isInitialized;
- (void)initialize;
@end

@implementation MagifyManager {
    MGFMagifyClient *_client;
}

static MagifyManager *_sharedInstance;
+ (instancetype)shared {
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        _sharedInstance = [[MagifyManager alloc] init];
    });
    return _sharedInstance;
}

- (BOOL)isInitialized {
    return _client != nil;
}

- (void)initialize {
    if (self.isInitialized) {
        NSLog(@"[Magify]: SDK already initialized.");
        return;
    }

    _client = [[MGFMagifyClient alloc] init];
    
    NSLog(@"[Magify]: Initialization complete.");
}

- (MGFMagifyClient *)getClient {
    return _client;
}

- (void)killClient {
    _client = nil;
}

@end

extern "C"
{
    typedef void (*CLOUD_STORE_VALUE_FETCHED_CALLBACK)(const char* key, const char* value);
    typedef void (*APPLICATION_STATE_CHANGED_CALLBACK)();

    void _magifyInitialize() {
        [[MagifyManager shared] initialize];
    }

    const char* _magifyGetDeviceModel() {
        return cStringCopy([[[MagifyManager shared] getClient] deviceModel]);
    }

    const char* _magifyGetOsVersion() {
        return cStringCopy([[[MagifyManager shared] getClient] osVersion]);
    }

    const char* _magifyGetBuildNumber() {
        return cStringCopy([[[MagifyManager shared] getClient] buildNumber]);
    }

    bool _magifyGetIsTablet() {
        return [[[MagifyManager shared] getClient] isTablet];
    }

    const char* _magifyGetLanguageWithRegion() {
        return cStringCopy([[[MagifyManager shared] getClient] languageWithRegion]);
    }

    const char* _magifyGetIdfa() {
        return cStringCopy([[[MagifyManager shared] getClient] idfa]);
    }

    const char* _magifyGetIdfv() {
        return cStringCopy([[[MagifyManager shared] getClient] idfv]);
    }

    const char* _magifyGetEncodedDeviceIdentifier() {
        return cStringCopy([[[MagifyManager shared] getClient] encodedDeviceIdentifier]);
    }

    void _magifySetOnCloudStoreValueFetched(CLOUD_STORE_VALUE_FETCHED_CALLBACK onCloudStoreValueFetched) {
        [[[MagifyManager shared] getClient] setOnCloudStoreValueFetched:^(NSString *key, NSString *value) {
            const char *cKey = [key UTF8String];
            const char *cValue = [value UTF8String];
            onCloudStoreValueFetched(cKey, cValue);
        }];
    }

    const char* _magifyGetStoreCountry() {
        return cStringCopy([[[MagifyManager shared] getClient] storeCountry]);
    }

    void _magifySetOnApplicationEnterForeground(APPLICATION_STATE_CHANGED_CALLBACK onApplicationEnterForeground) {
        [[[MagifyManager shared] getClient] setOnApplicationEnterForeground:^{
            onApplicationEnterForeground();
        }];
    }

    void _magifySetOnApplicationEnterBackground(APPLICATION_STATE_CHANGED_CALLBACK onApplicationEnterBackground) {
        [[[MagifyManager shared] getClient] setOnApplicationEnterBackground:^{
            onApplicationEnterBackground();
        }];
    }

    void _magifySetOnApplicationDidBecomeActive(APPLICATION_STATE_CHANGED_CALLBACK onApplicationDidBecomeActive) {
        [[[MagifyManager shared] getClient] setOnApplicationDidBecomeActive:^{
            onApplicationDidBecomeActive();
        }];
    }

    void _magifySetOnApplicationWillResignActive(APPLICATION_STATE_CHANGED_CALLBACK onApplicationWillResignActive) {
        [[[MagifyManager shared] getClient] setOnApplicationWillResignActive:^{
            onApplicationWillResignActive();
        }];
    }

    bool _magifyIsApplicationInstalled(const char* scheme) {
        NSString *schemeString = GetNullableStringParam(scheme);
        return [[[MagifyManager shared] getClient] isApplicationInstalledForScheme:schemeString];
    }

    void _magifyDispose() {
        [[[MagifyManager shared] getClient] dispose];
        [[MagifyManager shared] killClient];
    }

    const char* _magifyCollectMigrationData() {
        return cStringCopy([[[MagifyManager shared] getClient] collectMigrationData]);
    }

    void _magifyClearDataAfterMigration() {
        [[[MagifyManager shared] getClient] clearDataAfterMigration];
    }

    const char* _magifyGetStringValueInKeychain(const char* key) {
        NSString *keyString = GetNullableStringParam(key);
        return cStringCopy([[[MagifyManager shared] getClient] getStringValueInKeychainForKey:keyString]);
    }

    void _magifySetStringValueInKeychain(const char* value, const char* key) {
        NSString *valueString = GetNullableStringParam(value);
        NSString *keyString = GetNullableStringParam(key);
        [[[MagifyManager shared] getClient] setStringValueInKeychain:valueString forKey:keyString];
    }

    void _magifyRemoveValueInKeychain(const char* key) {
        NSString *keyString = GetNullableStringParam(key);
        [[[MagifyManager shared] getClient] removeValueInKeychainForKey:keyString];
    }

    void _magifyFetchValueFromCloudStore(const char* key) {
        NSString *keyString = GetNullableStringParam(key);
        [[[MagifyManager shared] getClient] fetchValueFromCloudStoreForKey:keyString];
    }

    void _magifySetValueToCloudStore(const char* value, const char* key) {
        NSString *valueString = GetNullableStringParam(value);
        NSString *keyString = GetNullableStringParam(key);
        [[[MagifyManager shared] getClient] setValueToCloudStore:valueString forKey:keyString];
    }

    void _magifyRemoveValueFromCloudStore(const char* key) {
        NSString *keyString = GetNullableStringParam(key);
        [[[MagifyManager shared] getClient] removeValueFromCloudStoreForKey:keyString];
    }

    void _magifyClearKeychains() {
        [[[MagifyManager shared] getClient] clearKeychains];
    }
}
