using System;
using System.Collections.Generic;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public class LimitedTimeOfferProvider : IDisposable
    {
        // TODO move this to PackageCallbacks??
        private readonly Subject<LtoInfo> _onUpdated = new Subject<LtoInfo>();
        private readonly Subject<LtoInfo> _onFinished = new Subject<LtoInfo>();
        private readonly List<LtoInfo> _activeOffers = new();
        private readonly CompositeDisposable _disposables = new();

        public IObservable<LtoInfo> OnUpdated => _onUpdated;
        public IObservable<LtoInfo> OnFinished => _onFinished;
        public IReadOnlyList<LtoInfo> ActiveOffers => _activeOffers;

        public LimitedTimeOfferProvider()
        {
            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnAdded += handler, handler => MagifyManager.Lto.OnAdded -= handler)
                .Subscribe(AddedOffer)
                .AddTo(_disposables);

            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnRemoved += handler, handler => MagifyManager.Lto.OnRemoved -= handler)
                .Subscribe(RemovedOffer)
                .AddTo(_disposables);

            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnUpdated += handler, handler => MagifyManager.Lto.OnUpdated -= handler)
                .Subscribe(UpdatedOffer)
                .AddTo(_disposables);

            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnFinished += handler, handler => MagifyManager.Lto.OnFinished -= handler)
                .Subscribe(FinishedOffer)
                .AddTo(_disposables);
        }

        public void CompleteActiveLto(string offerName)
        {
            MagifyManager.Lto.CompleteActiveLto(offerName);
        }

        internal async UniTaskVoid HandleOfferEventsInNextFrameStart([NotNull] LimitedTimeOfferNativeEventsAccumulator offerNativeEventsAccumulator)
        {
            await UniTask.NextFrame(PlayerLoopTiming.PreUpdate);
            offerNativeEventsAccumulator.SpendCurrentEvents(AddedOffer, RemovedOffer, UpdatedOffer, FinishedOffer);
        }

        private void AddedOffer(LtoInfo ltoInfo)
        {
            if (NeedIgnoreOffer(ltoInfo)) return;

            _activeOffers.Add(ltoInfo);
        }

        private void RemovedOffer(LtoInfo ltoInfo)
        {
            if (NeedIgnoreOffer(ltoInfo)) return;

            _activeOffers.Remove(ltoInfo);
        }

        private void UpdatedOffer(LtoInfo ltoInfo)
        {
            if (NeedIgnoreOffer(ltoInfo)) return;

            _onUpdated.OnNext(ltoInfo);
        }

        private void FinishedOffer(LtoInfo ltoInfo)
        {
            _onFinished.OnNext(ltoInfo);
        }

        private bool NeedIgnoreOffer(LtoInfo ltoInfo)
        {
            return ltoInfo.BadgeCreative.Resource.Type != CreativeResource.ArtifactType.Image;
        }

        void IDisposable.Dispose()
        {
            _disposables.Dispose();
        }
    }
}