using System;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class BasePurchaserService : IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [CanBeNull]
        public IMinimalInAppStore InAppStore { get; private set; }
        [NotNull]
        public IReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler => MagifyManager.ExternalPurchaseVerificationHandler;
        [NotNull]
        public IReactiveProperty<float> VerificationRetryInterval => MagifyManager.VerificationRetryInterval;

        public virtual void SetPurchasingProvider([CanBeNull] IMinimalInAppStore inAppStore)
        {
            DisposeCurrentInAppStore();
            InAppStore = inAppStore;
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished += ExternalPurchaseFinishedHandler;
            }
        }

        /// <inheritdoc cref="MagifyManager.TrackTrustedPurchase"/>
        public void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record)
        {
            if (record.Type is PurchaseType.InitialPurchase)
            {
                _logger.LogWarning($"{nameof(TrackTrustedPurchase)} has been called with {nameof(TrustedPurchaseRecord)}.{nameof(TrustedPurchaseRecord.Type)}:{nameof(PurchaseType.InitialPurchase)}."
                        + "It's not recommended way to track this type of trusted purchases."
                        + $"You should check {nameof(TrackInApp)} or {nameof(BaseSubscriptionService.ProductFetchedHandler)} for basic integration,"
                        + $"and {nameof(CampaignRequest.TrackObtain)} or {nameof(IMinimalInAppStore.OnPurchaseFinished)} for advanced integration."
                        + "This invocation is going to handle trusted purchase record, but be careful with trusted purchases tracking.");
            }
            MagifyManager.TrackTrustedPurchase(record);
        }

        private void DisposeCurrentInAppStore()
        {
            if (InAppStore != null)
            {
                InAppStore.OnPurchaseFinished -= ExternalPurchaseFinishedHandler;
            }
        }

        void IDisposable.Dispose()
        {
            DisposeCurrentInAppStore();
        }

        private void ExternalPurchaseFinishedHandler([NotNull] string productId, [NotNull] PurchaseInfo purchaseInfo)
        {
            if (purchaseInfo.SubscriptionInfo == null)
            {
                TrackInApp(productId, purchaseInfo);
            }
        }

        private void TrackInApp([NotNull] string productId, [NotNull] PurchaseInfo purchaseInfo)
        {
            if (purchaseInfo.TrustedPurchaseRecord != null)
            {
                MagifyManager.TrackExternalTrustedInAppFor(purchaseInfo.TrustedPurchaseRecord);
            }
            else if (purchaseInfo.SkipVerification)
            {
                MagifyManager.TrackExternalInAppWithoutVerification(
                    productId,
                    purchaseInfo.Price,
                    purchaseInfo.Currency,
                    purchaseInfo.TransactionId,
                    purchaseInfo.OriginalTransactionId,
                    purchaseInfo.Store);
            }
            else
            {
                MagifyManager.TrackExternalInAppFor(
                    productId,
                    purchaseInfo.Price,
                    purchaseInfo.Currency,
                    purchaseInfo.TransactionId,
                    purchaseInfo.PurchaseToken,
                    purchaseInfo.OriginalTransactionId,
                    purchaseInfo.Receipt?.Payload,
                    purchaseInfo.Store);
            }
        }
    }
}