using System;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    [Serializable]
    public abstract class RemoteProperty<T> : IReadOnlyReactiveProperty<T>
    {
        [SerializeField]
        private string _name;
        [SerializeField]
        private T _defaultValue;

        public bool HasValue => true;

        public T Value => TryGetFeature(_name, out var feature)
                ? feature.Value
                : _defaultValue;

        protected RemoteProperty(string name, T defaultValue)
        {
            _name = name;
            _defaultValue = defaultValue;
        }

        public IDisposable Subscribe(IObserver<T> observer)
        {
            return Subscribe(_name, observer);
        }

        protected abstract bool TryGetFeature(string name, out Feature<T> value);
        public abstract IDisposable Subscribe(string name, IObserver<T> observer);
        public static implicit operator T(RemoteProperty<T> property) => property.Value;
    }

    [Serializable]
    public class NumberRemoteProperty : RemoteProperty<double>
    {
        public NumberRemoteProperty(string name, double defaultValue) : base(name, defaultValue)
        {
        }
        public NumberRemoteProperty(string name, float defaultValue) : base(name, defaultValue)
        {
        }

        protected override bool TryGetFeature(string name, out Feature<double> value)
        {
            value = default;
            return MagifyService.Instance != null && MagifyService.Instance.Features.TryGetNumber(name, out value);
        }

        public override IDisposable Subscribe(string name, IObserver<double> observer)
        {
            MagifyService.ThrowIfNotReady();
            return MagifyService.Instance.Features.GetReactiveNumber(name).Subscribe(observer);
        }
    }

    [Serializable]
    public class StringRemoteProperty : RemoteProperty<string>
    {
        public StringRemoteProperty(string name, string defaultValue) : base(name, defaultValue)
        {
        }

        protected override bool TryGetFeature(string name, out Feature<string> value)
        {
            value = default;
            return MagifyService.Instance != null && MagifyService.Instance.Features.TryGetString(name, out value);
        }

        public override IDisposable Subscribe(string name, IObserver<string> observer)
        {
            MagifyService.ThrowIfNotReady();
            return MagifyService.Instance.Features.GetReactiveString(name).Subscribe(observer);
        }
    }

    [Serializable]
    public class BoolRemoteProperty : RemoteProperty<bool>
    {
        public BoolRemoteProperty(string name, bool defaultValue) : base(name, defaultValue)
        {
        }

        protected override bool TryGetFeature(string name, out Feature<bool> value)
        {
            value = default;
            return MagifyService.Instance != null && MagifyService.Instance.Features.TryGetBool(name, out value);
        }

        public override IDisposable Subscribe(string name, IObserver<bool> observer)
        {
            MagifyService.ThrowIfNotReady();
            return MagifyService.Instance.Features.GetReactiveBool(name).Subscribe(observer);
        }
    }
}