﻿#if UNITY_PURCHASES
using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using UnityEngine.Purchasing;

namespace Magify
{
    public static class UnityPurchasesUtils
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        /// <remarks>
        /// Use it if you are SURE, that Unity Purchases already Initialized. It is important to remember, that some Stores have a specific logic on app minimize-maximize. Usually it's better to use this method only on application start.
        /// </remarks>
        public static SubscriptionInfo LoadSubscriptionInfo(string productId, IStoreController controller, IExtensionProvider extensions)
        {
            var unitySubscriptionModel = GetUnitySubscriptionModel(productId, controller, extensions);
            if (unitySubscriptionModel == null) return default;
            var expireDate = unitySubscriptionModel.GetExpireDate();
            if (MagifyManager.Environment == Environment.Staging
             && Application.platform == RuntimePlatform.Android
             && unitySubscriptionModel.IsExpired() == Result.True && expireDate > DateTime.UtcNow)
            {
                // In google play (test mode) expire date may be incorrect, when product is expired.
                expireDate = DateTime.UtcNow - TimeSpan.FromSeconds(1);
            }
            return new SubscriptionInfo(productId, unitySubscriptionModel.IsFreeTrial() == Result.True, expireDate, unitySubscriptionModel.GetSubscriptionPeriod());
        }

        public static UnitySubscriptionModel GetUnitySubscriptionModel(string productId, IStoreController controller, IExtensionProvider extensions)
        {
            if (Application.isEditor) return new UnitySubscriptionModel(productId);
            var storeProduct = controller.products.all.FirstOrDefault(p => p.definition.storeSpecificId == productId && p.hasReceipt);
            if (storeProduct == null)
            {
                _logger.Log($"Cannot load subscription product from UnityPurchasing for product {productId}.");
                return null;
            }
            if (storeProduct.definition.type != UnityEngine.Purchasing.ProductType.Subscription)
            {
                _logger.Log($"Cannot load subscription product from UnityPurchasing for product {productId} it's not a subscription.");
                return null;
            }
            var subscriptions = Application.platform switch
            {
                RuntimePlatform.IPhonePlayer => extensions.GetExtension<IAppleExtensions>().GetProductDetails(),
                RuntimePlatform.Android => controller.products.all.ToDictionary(productN => productN.definition.storeSpecificId, productN => productN.metadata.GetGoogleProductMetadata()?.originalJson),
                _ => new Dictionary<string, string>(),
            };
            subscriptions.TryGetValue(productId, out var subscriptionMetadataJson);

            var subscriptionInfo = new SubscriptionManager(storeProduct, subscriptionMetadataJson).getSubscriptionInfo();
            return new UnitySubscriptionModel(storeProduct, subscriptionMetadataJson, subscriptionInfo);
        }
    }
}
#endif