﻿#if MAX_ADS_MEDIATOR
using System;
using System.Diagnostics.CodeAnalysis;
using UnityEngine;

namespace Magify
{
    [SuppressMessage("ReSharper", "AccessToStaticMemberViaDerivedType")]
    public class MaxMediator : IAdsMediator
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        #region BannerEvents

        public event Action<string> OnBannerLoadFailed;
        public event Action<IAdsImpression> OnBannerLoaded;
        public event Action<IAdsImpression> OnBannerClicked;

        #endregion

        #region RewardEvents

        public event Action<IAdsImpression> OnRewardShown;
        public event Action OnRewardLoaded;
        public event Action<string> OnRewardLoadFailed;
        public event Action<string> OnRewardDisplayFailed;
        public event Action OnRewardReceived;
        public event Action<IAdsImpression> OnRewardClicked;
        public event Action<IAdsImpression> OnRewardHidden;

        #endregion

        #region InterEvents

        public event Action<IAdsImpression> OnInterShown;
        public event Action OnInterLoaded;
        public event Action<string> OnInterLoadFailed;
        public event Action<string> OnInterDisplayFailed;
        public event Action<IAdsImpression> OnInterClicked;
        public event Action<IAdsImpression> OnInterHidden;

        #endregion

        private readonly MagifySettings _settings;
        private MaxSdkBase.SdkConfiguration _configuration;

        private MaxSdkBase.AdInfo _rewardInfo;
        private MaxSdkBase.AdInfo _interInfo;

        public bool IsInitialized => MaxSdk.IsInitialized();
        public bool IsBannerVisible { get; private set; }
        public bool IsRewardedVideoReady => _rewardInfo != null && MaxSdk.IsRewardedAdReady(RewardAdUnitId);
        public bool IsInterVideoReady => _interInfo != null && MaxSdk.IsInterstitialReady(InterAdUnitId);

        public MaxSdkBase.SdkConfiguration Configuration => _configuration;
        public string InterAdUnitId { get; private set; }
        public string RewardAdUnitId { get; private set; }
        public string BannerAdUnitId { get; private set; }

        public MaxMediator(MagifySettings settings)
        {
            _settings = settings;
        }

        public void Initialize()
        {
            switch (Application.platform)
            {
                case RuntimePlatform.Android:
                    InterAdUnitId = _settings.InterAdUnitIdGP;
                    RewardAdUnitId = _settings.RewardAdUnitIdGP;
                    BannerAdUnitId = _settings.BannerAdUnitIdGP;
                    break;
                case RuntimePlatform.IPhonePlayer:
                    InterAdUnitId = _settings.InterAdUnitIdIOS;
                    RewardAdUnitId = _settings.RewardAdUnitIdIOS;
                    BannerAdUnitId = _settings.BannerAdUnitIdIOS;
                    break;
                case RuntimePlatform.LinuxEditor:
                case RuntimePlatform.WindowsEditor:
                case RuntimePlatform.OSXEditor:
                    InterAdUnitId = _settings.InterAdUnitIdGP;
                    RewardAdUnitId = _settings.RewardAdUnitIdGP;
                    BannerAdUnitId = _settings.BannerAdUnitIdGP;
                    break;
            }

            _logger.Log($"{nameof(InterAdUnitId)}={InterAdUnitId}");
            _logger.Log($"{nameof(RewardAdUnitId)}={RewardAdUnitId}");
            _logger.Log($"{nameof(_settings.SdkKey)}={_settings.SdkKey}");

            MaxSdk.SetSdkKey(_settings.SdkKey);
            MaxSdk.SetUserId(MagifyManager.ClientId);
            MaxSdk.SetExtraParameter("return_audio_focus", "true");

            MaxSdkCallbacks.OnSdkInitializedEvent += OnSdkInitialized;

            MaxSdkCallbacks.Banner.OnAdClickedEvent += BannerAdClickedHandler;
            MaxSdkCallbacks.Banner.OnAdLoadedEvent += BannerAdLoadedHandler;
            MaxSdkCallbacks.Banner.OnAdLoadFailedEvent += BannerAdFailedHandler;

            MaxSdkCallbacks.Rewarded.OnAdDisplayedEvent += RaiseRewardDisplayed;
            MaxSdkCallbacks.Rewarded.OnAdLoadedEvent += RaiseRewardLoaded;
            MaxSdkCallbacks.Rewarded.OnAdLoadFailedEvent += RaiseRewardLoadFailed;
            MaxSdkCallbacks.Rewarded.OnAdDisplayFailedEvent += RaiseRewardDisplayFailed;
            MaxSdkCallbacks.Rewarded.OnAdReceivedRewardEvent += RaiseRewardReceived;
            MaxSdkCallbacks.Rewarded.OnAdClickedEvent += RaiseRewardClicked;
            MaxSdkCallbacks.Rewarded.OnAdHiddenEvent += RaiseRewardHidden;

            MaxSdkCallbacks.Interstitial.OnAdDisplayedEvent += RaiseInterDisplayed;
            MaxSdkCallbacks.Interstitial.OnAdLoadedEvent += RaiseInterLoaded;
            MaxSdkCallbacks.Interstitial.OnAdLoadFailedEvent += RaiseInterLoadFailed;
            MaxSdkCallbacks.Interstitial.OnAdDisplayFailedEvent += RaiseInterDisplayFailed;
            MaxSdkCallbacks.Interstitial.OnAdClickedEvent += RaiseInterClicked;
            MaxSdkCallbacks.Interstitial.OnAdHiddenEvent += RaiseInterHidden;

            MaxSdk.InitializeSdk();
        }

        private void OnSdkInitialized(MaxSdkBase.SdkConfiguration configuration)
        {
            _configuration = configuration;
            MaxSdkCallbacks.OnSdkInitializedEvent -= OnSdkInitialized;
        }

        public void Terminate()
        {
            _logger.Log("Terminate Max");

            _interInfo = null;
            _rewardInfo = null;

            MaxSdkCallbacks.Banner.OnAdClickedEvent -= BannerAdClickedHandler;
            MaxSdkCallbacks.Banner.OnAdLoadedEvent -= BannerAdLoadedHandler;
            MaxSdkCallbacks.Banner.OnAdLoadFailedEvent -= BannerAdFailedHandler;

            MaxSdkCallbacks.Rewarded.OnAdDisplayedEvent -= RaiseRewardDisplayed;
            MaxSdkCallbacks.Rewarded.OnAdLoadedEvent -= RaiseRewardLoaded;
            MaxSdkCallbacks.Rewarded.OnAdLoadFailedEvent -= RaiseRewardLoadFailed;
            MaxSdkCallbacks.Rewarded.OnAdDisplayFailedEvent -= RaiseRewardDisplayFailed;
            MaxSdkCallbacks.Rewarded.OnAdReceivedRewardEvent -= RaiseRewardReceived;
            MaxSdkCallbacks.Rewarded.OnAdClickedEvent -= RaiseRewardClicked;
            MaxSdkCallbacks.Rewarded.OnAdHiddenEvent -= RaiseRewardHidden;

            MaxSdkCallbacks.Interstitial.OnAdDisplayedEvent -= RaiseInterDisplayed;
            MaxSdkCallbacks.Interstitial.OnAdLoadedEvent -= RaiseInterLoaded;
            MaxSdkCallbacks.Interstitial.OnAdLoadFailedEvent -= RaiseInterLoadFailed;
            MaxSdkCallbacks.Interstitial.OnAdDisplayFailedEvent -= RaiseInterDisplayFailed;
            MaxSdkCallbacks.Interstitial.OnAdClickedEvent -= RaiseInterClicked;
            MaxSdkCallbacks.Interstitial.OnAdHiddenEvent -= RaiseInterHidden;
        }

        public void SetUserConsent(bool consent)
        {
            MaxSdk.SetHasUserConsent(consent);
        }

        public void ShowMediationDebugger()
        {
            MaxSdk.ShowMediationDebugger();
        }

        #region Banner

        public void ShowBanner()
        {
            if (string.IsNullOrEmpty(BannerAdUnitId))
            {
                _logger.Log($"{nameof(ShowBanner)} - skip because {nameof(BannerAdUnitId)} is empty");
                return;
            }

            _logger.Log($"{nameof(ShowBanner)} - create, setup and show banner with {nameof(BannerAdUnitId)}={BannerAdUnitId}");
            MaxSdk.CreateBanner(BannerAdUnitId, MaxSdkBase.BannerPosition.BottomCenter);
            MaxSdk.SetBannerExtraParameter(BannerAdUnitId, "adaptive_banner", "false");
            MaxSdk.SetBannerBackgroundColor(BannerAdUnitId, new Color(0, 0, 0, 0));
            MaxSdk.ShowBanner(BannerAdUnitId);
        }

        public void HideBanner()
        {
            if (string.IsNullOrEmpty(BannerAdUnitId))
            {
                _logger.Log($"{nameof(HideBanner)} called - skip it because {nameof(BannerAdUnitId)} is empty");
                return;
            }

            _logger.Log($"{nameof(ShowBanner)} - hide and destroy banner with {nameof(BannerAdUnitId)}={BannerAdUnitId}");
            MaxSdk.HideBanner(BannerAdUnitId);
            MaxSdk.DestroyBanner(BannerAdUnitId);
            IsBannerVisible = false;
        }

        private void BannerAdClickedHandler(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Banner with id {id} clicked");
            OnBannerClicked?.Invoke(GetImpression(info));
        }

        private void BannerAdLoadedHandler(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Banner with id {id} expanded");
            IsBannerVisible = true;
            OnBannerLoaded?.Invoke(GetImpression(info));
        }

        private void BannerAdFailedHandler(string id, MaxSdkBase.ErrorInfo info)
        {
            _logger.LogError($"Banner with id {id} failed to display:{System.Environment.NewLine}{info}");
            IsBannerVisible = false;
            OnBannerLoadFailed?.Invoke(info.Message);
        }

        #endregion

        public void LoadRewardVideo()
        {
            _rewardInfo = null;
            MaxSdk.LoadRewardedAd(RewardAdUnitId);
        }

        public void LoadInterVideo()
        {
            _interInfo = null;
            MaxSdk.LoadInterstitial(InterAdUnitId);
        }

        public void ShowRewardVideo(string placement)
        {
            MaxSdk.ShowRewardedAd(RewardAdUnitId, placement);
        }

        public void ShowInterVideo(string placement)
        {
            MaxSdk.ShowInterstitial(InterAdUnitId, placement);
        }

        public IAdsImpression GetImpression(MaxSdkBase.AdInfo info)
        {
            return new ApplovinAdsImpression
            {
                Id = info.AdUnitIdentifier,
                Network = info.NetworkName,
                Precision = info.RevenuePrecision,
                Revenue = info.Revenue
                // We don't have ImpressionId then leave it empty
            };
        }

        #region Raise reward events

        protected virtual void RaiseRewardDisplayed(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Reward video ad with id {id} has been shown, ad info: \n{info}");
            _rewardInfo = info;
            OnRewardShown?.Invoke(GetImpression(info));
        }

        private void RaiseRewardLoaded(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Reward video - {id} loaded. NetworkName: {info.NetworkName}; AdUnitIdentifier: {info.AdUnitIdentifier}; CreativeIdentifier: {info.CreativeIdentifier}");
            _rewardInfo = info;
            OnRewardLoaded?.Invoke();
        }

        private void RaiseRewardLoadFailed(string id, MaxSdkBase.ErrorInfo error)
        {
            _logger.LogError($"RewardVideo - Can't load reward video {id}. Error: {error.Message}");
            OnRewardLoadFailed?.Invoke(error.ToString());
        }

        private void RaiseRewardDisplayFailed(string id, MaxSdkBase.ErrorInfo error, MaxSdkBase.AdInfo info)
        {
            _logger.LogError($"Failed to show RewardVideo with id {id}");
            OnRewardDisplayFailed?.Invoke(error.ToString());
        }

        private void RaiseRewardReceived(string id, MaxSdkBase.Reward reward, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"RewardVideo with id {id} rewarded");
            OnRewardReceived?.Invoke();
        }

        private void RaiseRewardClicked(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"RewardVideo with id {id} clicked");
            OnRewardClicked?.Invoke(GetImpression(info));
        }

        private void RaiseRewardHidden(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"RewardVideo with id {id} hidden");
            OnRewardHidden?.Invoke(GetImpression(info));
        }

        #endregion

        #region Raise inter events

        protected virtual void RaiseInterDisplayed(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Interstitial ad with id {id} has been shown, ad info: \n{info}");
            _interInfo = info;
            OnInterShown?.Invoke(GetImpression(info));
        }

        private void RaiseInterLoaded(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"Interstitial video {id} loaded. NetworkName: {info.NetworkName}; AdUnitIdentifier: {info.AdUnitIdentifier}; CreativeIdentifier: {info.CreativeIdentifier}");
            _interInfo = info;
            OnInterLoaded?.Invoke();
        }

        private void RaiseInterLoadFailed(string id, MaxSdkBase.ErrorInfo error)
        {
            _logger.LogError($"InterVideo - Can't load interstitial video {id}. Error: {error.Message}");
            OnInterLoadFailed?.Invoke(error.ToString());
        }

        private void RaiseInterDisplayFailed(string id, MaxSdkBase.ErrorInfo error, MaxSdkBase.AdInfo info)
        {
            _logger.LogError($"Failed to show InterVideo with id {id}");
            OnInterDisplayFailed?.Invoke(error.ToString());
        }

        private void RaiseInterClicked(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"InterVideo with id {id} clicked");
            OnInterClicked?.Invoke(GetImpression(info));
        }

        private void RaiseInterHidden(string id, MaxSdkBase.AdInfo info)
        {
            _logger.Log($"InterVideo with id {id} hidden");
            OnInterHidden?.Invoke(GetImpression(info));
        }

        #endregion
    }
}
#endif