﻿using JetBrains.Annotations;
using UnityEditor;
using UnityEngine;

namespace Magify
{
    [CustomPropertyDrawer(typeof(MagifySettingsOptionalAttribute))]
    internal class MagifySettingsOptionalAttributeEditor : PropertyDrawer
    {
#if MAGIFY_ADVANCED
        private const bool HideAdvanced = false;
#else
        private const bool HideAdvanced = true;
#endif

        private const float HeaderSpaceHeight = 16;
        private static readonly float _headerHeight = EditorGUIUtility.singleLineHeight;

        public override void OnGUI(Rect position, SerializedProperty property, GUIContent label)
        {
            using (property)
            {
                if (fieldInfo!.IsStatic) return;
                var isArray = fieldInfo.FieldType.IsArray;

                EditorGUI.BeginProperty(position, label, property);
                if (!HideAdvanced && !isArray)
                {
                    if (TryGetHeader(out var header))
                    {
                        EditorGUI.LabelField(position, header, EditorStyles.boldLabel);
                        position.yMin += _headerHeight + HeaderSpaceHeight;
                        EditorGUI.PropertyField(position, property, label, includeChildren: true);
                    }
                    else
                    {
                        EditorGUI.PropertyField(position, property, label, includeChildren: true);
                    }
                }

                if (isArray)
                {
                    EditorGUI.PropertyField(position, property, label, includeChildren: true);
                }
                EditorGUI.EndProperty();
            }
        }

        public override float GetPropertyHeight(SerializedProperty property, GUIContent label)
        {
            if (!HideAdvanced && !fieldInfo!.FieldType.IsArray
             || fieldInfo!.FieldType.IsArray)
            {
                return EditorGUI.GetPropertyHeight(property) + (TryGetHeader(out _) ? _headerHeight + HeaderSpaceHeight : 0);
            }
            return -EditorGUIUtility.standardVerticalSpacing;
        }

        [ContractAnnotation("=> false, header:null; => true, header:notnull")]
        private bool TryGetHeader([CanBeNull] out string header)
        {
            header = ((MagifySettingsOptionalAttribute)attribute)?.Header;
            return header != null;
        }
    }
}