#if UNITY_LOCALIZATION
using System.Globalization;
using NUnit.Framework;
using UnityEditor.Localization;
using UnityEngine.Localization.Settings;
using UnityEngine.Localization.SmartFormat;
using UnityEngine.Localization.Tables;

namespace Magify.BundleCreator.Editor.Tests
{
    public class FontCustomizerTests
    {
        [SetUp]
        public void InitializeLocalization()
        {
            LocalizationSettings.InitializationOperation.WaitForCompletion();
        }

        [Test]
        public void WhenGetGetUsedLocalizedCharacters_AndStringIsSmart_ThenShouldReturnAllPossibleCharacters()
        {
            // Arrange.
            // Act.
            var usedLocalizedCharacters = BundleFontCustomizer.GetUsedLocalizedCharacters("en", "booster.cell.farm");

            // Assert.
            Assert.AreEqual("Cels", usedLocalizedCharacters);
        }

        [TestCase("en", "{0:p:Cell|Cells}", 0, "Cells")]
        [TestCase("en", "{0:p:Cell|Cells}", 1, "Cell")]
        [TestCase("en", "{0:p:Cell|Cells}", 123, "Cells")]
        [TestCase("ru", "{0:p:Таймер|Таймера|Таймеров|Таймеров}", 1, "Таймер")]
        [TestCase("ru", "{0:p:Таймер|Таймера|Таймеров|Таймеров}", 2, "Таймера")]
        [TestCase("ru", "{0:p:Таймер|Таймера|Таймеров|Таймеров}", 10, "Таймеров")]
        public void WhenFormatLocalization_AndLocalizationIsPlural_ThenShouldReturnCorrectLocal(string cultureCode, string format, int value, string expected)
        {
            // Arrange.
            // Act.
            var localization = Smart.Format(CultureInfo.GetCultureInfo(cultureCode), format, value);

            // Assert.
            Assert.AreEqual(expected, localization);
        }

        [TestCase("booster.cell.farm", "ru", 1, "Ячейка")]
        [TestCase("booster.cell.farm", "en", 1, "Cell")]
        public void WhenGetLocalizedString_AndHasLocalization_ThenShouldReturnCorrectLocalization(string key, string cultureCode, int value, string expected)
        {
            // Arrange.
            var localizationCollection = LocalizationEditorSettings.GetStringTableCollection(UnityLocalizerTables.GlobalLocalizationTableName);
            var currentCultureTable = localizationCollection.GetTable(cultureCode) as StringTable;
            var localizeEntry = currentCultureTable.GetEntry(key);

            // Act.
            var localizedString = localizeEntry.GetLocalizedString(value);

            // Assert.
            Assert.AreEqual(expected, localizedString);
        }
    }
}
#endif