using System;
using System.Linq;
using FluentAssertions;
using NUnit.Framework;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify.Tests
{
    internal class WhereTests
    {
        [NotNull, ItemNotNull]
        private static object[] _predicates = {
            new object[] { (Func<object, bool>)(n => (int)n > 0), 1, -1 },
            new object[] { (Func<object, bool>)(n => (string)n == "a"), "a", "b" },
            new object[] { (Func<object, bool>)(n => (bool)n == true), true, false },
            new object[] { (Func<object, bool>)(n => n != null), new object(), null },
        };

        [NotNull, ItemNotNull]
        private static object[] _manyPredicates = _predicates.Select(o => (object[])o).Select(arr => arr.Append(100).ToArray()).ToArray();

        [Test]
        [TestCaseSource(nameof(_predicates))]
        public void WhenWhereApplied_AndNext_ThenCallback([NotNull] Func<object, bool> predicate, [CanBeNull] object correct, [CanBeNull] object wrong)
        {
            // Arrange
            var received = 0;
            using var subject = new Subject<object>();
            using var _ = subject.Where(predicate).Subscribe(_ => received++);

            // Act
            subject.OnNext(correct);
            subject.OnNext(wrong);

            // Assert
            received.Should()!.Be(1);
        }

        [Test]
        [TestCaseSource(nameof(_manyPredicates))]
        public void WhenWhereApplied_AndManyNext_ThenManyCallbacks([NotNull] Func<object, bool> predicate, [CanBeNull] object correct, [CanBeNull] object wrong, int repeats)
        {
            // Arrange
            var received = 0;
            using var subject = new Subject<object>();
            using var _ = subject.Where(predicate).Subscribe(_ => received++);

            // Act
            repeats.EnumerateEach().ForEach(_ =>
            {
                subject.OnNext(correct);
                subject.OnNext(wrong);
            });

            // Assert
            received.Should()!.Be(repeats);
        }
    }
}