using System;
using System.Linq;
using System.Collections.Generic;
using FluentAssertions;
using NUnit.Framework;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify.Tests
{
    internal class WhereSelectTests
    {
        [NotNull, ItemNotNull]
        private static readonly object[] _whereSelect =
        {
            new object[] { (Func<object, bool>)(n => (int)n! > 0), 2, -1, (Func<object, object>)(n => (int)n! * (int)n), 4 },
            new object[] { (Func<object, bool>)(s => (string)s == "a"), "a", "b", (Func<object, object>)(s => (string)s + "b"), "ab" },
        };

        private static readonly object[] _manyWhereSelect = _whereSelect.Select(arr => ((object[])arr).Append(100).ToArray()).ToArray();

        [Test]
        [TestCaseSource(nameof(_whereSelect))]
        public void WhenWhereApplied_AndNext_ThenCallback(
                [NotNull] Func<object, bool> predicate, [CanBeNull] object correct, [CanBeNull] object wrong,
                [NotNull] Func<object, object> selector, [CanBeNull] object expected)
        {
            // Arrange
            var selected = new List<object>();
            using var subject = new Subject<object>();
            using var _ = subject.Where(predicate).Select(selector).Subscribe(s => selected.Add(s));

            // Act
            subject.OnNext(correct);
            subject.OnNext(wrong);

            // Assert
            selected.Count.Should()!.Be(1);
            selected.Should()!.Contain(expected);
        }

        [Test]
        [TestCaseSource(nameof(_manyWhereSelect))]
        public void WhenWhereApplied_AndManyNext_ThenManyCallbacks(
                [NotNull] Func<object, bool> predicate, [CanBeNull] object correct, [CanBeNull] object wrong,
                [NotNull] Func<object, object> selector, [CanBeNull] object expected,
                int repeats)
        {
            // Arrange
            var selected = new List<object>(repeats);
            using var subject = new Subject<object>();
            using var _ = subject.Where(predicate).Select(selector).Subscribe(s => selected.Add(s));

            // Act
            repeats.EnumerateEach().ForEach(_ =>
            {
                subject.OnNext(correct);
                subject.OnNext(wrong);
            });

            // Assert
            selected.Count.Should()!.Be(repeats);
            selected.Should()!.AllBeEquivalentTo(expected);
        }
    }
}