using System;
using System.Diagnostics.CodeAnalysis;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class Context : ServerApiTestsPart
        {
            [SuppressMessage("ReSharper", "PossibleNullReferenceException")]
            private static object[] MultiplyConfigScopesCases =>
                new object[]
                {
                    new object[] { ConfigScope.Limits | ConfigScope.ProductIds, (Func<CampaignsContext, bool>)(context =>
                    {
                        return context.Limits is null && context.DefaultProducts is null;
                    }) },
                    new object[] { ConfigScope.Limits | ConfigScope.ProductIds | ConfigScope.Campaigns | ConfigScope.AbTests, (Func<CampaignsContext, bool>)(context =>
                    {
                        return context.Limits is null && context.DefaultProducts is null && context.CampaignModels is null && context.AssignedAbTests is null;
                    }) },
                    new object[] { ConfigScope.Limits | ConfigScope.ProductIds | ConfigScope.Content | ConfigScope.StoredAppFeatures | ConfigScope.AbTests | ConfigScope.Campaigns, (Func<CampaignsContext, bool>)(context =>
                    {
                        return context.Limits is null && context.DefaultProducts is null && context.StoredAppFeatures is null && context.AssignedAbTests is null && context.Content is null && context.CampaignModels is null;
                    }) },
                };

            [SuppressMessage("ReSharper", "PossibleNullReferenceException")]
            private static object[] ConfigScopesCases =>
                new object[]
                {
                    new object[] { ConfigScope.Limits, (Func<CampaignsContext, bool>)(context => context.Limits is null) },
                    new object[] { ConfigScope.ProductIds, (Func<CampaignsContext, bool>)(context => context.DefaultProducts is null) },
                    new object[] { ConfigScope.Content, (Func<CampaignsContext, bool>)(context => context.Content is null) },
                    new object[] { ConfigScope.Campaigns, (Func<CampaignsContext, bool>)(context => context.CampaignModels is null) },
                    new object[] { ConfigScope.AppFeatures, (Func<CampaignsContext, bool>)(context => context.Features is null) },
                    new object[] { ConfigScope.Segmentations, (Func<CampaignsContext, bool>)(context => context.Segmentations is null) },
                    new object[] { ConfigScope.AbTests, (Func<CampaignsContext, bool>)(context => context.AssignedAbTests is null) },
                    new object[] { ConfigScope.StoredAppFeatures, (Func<CampaignsContext, bool>)(context => context.StoredAppFeatures is null) },
                };

            [Test]
            public async Task WhenContextRequested_ThenIsContextRequestFlagShouldBeTrue()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                var isContextRequest = false;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                UniTask.RunOnThreadPool(async () =>
                {
                    await UniTask.Delay(2, cancellationToken: cancellationToken);
                    isContextRequest = serverApi.IsContextRequest;
                }, cancellationToken: cancellationToken).Forget();
                await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);

                // Assert
                isContextRequest.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(3)]
            [TestCase(5)]
            [TestCase(10)]
            public void WhenContextRequested_AndGetExpiredErrorCode_ThenRequestShouldBeRetry(int expectedFailed)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(100000);
                var retires = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (retires < expectedFailed)
                    {
                        retires++;
                        return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse { Error = new() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    _ = serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                    UniTask.Delay(100, cancellationToken : cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                retires.Should()!.Be(expectedFailed);
            }

            [Test]
            public void WhenContextRequested_AndGetExpiredErrorCode_ThenShouldBeRequestForToken()
            {
                // Arrange
                var cancellationToken = GetCancellationToken(100000);
                var firstGetTokenReceived = false;
                var secondGetTokenReceived = false;
                var getContextReceived = false;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (message.Method is WebRequestMethods.GetToken or WebRequestMethods.ReissueToken)
                    {
                        firstGetTokenReceived = true;
                        if (getContextReceived) secondGetTokenReceived = true;
                    }
                    else if (message.Method is WebRequestMethods.GetContext)
                    {
                        getContextReceived = true;
                         return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new ErrorResponse.ErrorInfo() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    _ = serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                    UniTask.Delay(100, cancellationToken : cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                firstGetTokenReceived.Should()!.BeTrue();
                secondGetTokenReceived.Should()!.BeTrue();
                getContextReceived.Should()!.BeTrue();
            }

            [Test]
            public async Task WhenContextRequested_AndCancelAllTokenLoadings_ThenNoOperationCancelledException()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await serverApi.CancelAllTokenLoadings();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.BeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenContextRequested_AndCancelAllContextLoadings_ThenGetOperationCancelledException()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await serverApi.CancelAllContextLoadings();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(100)]
            public async Task WhenMultiplyLoadingContext_ThenRequestToNetworkShouldBeOne(int loadingsNumber)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var shouldCountRequest = false;
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (shouldCountRequest) requestCount++;
                    promise.TrySetResult();
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    shouldCountRequest = true;
                    loadingsNumber.EnumerateEach().ForEach(_ => serverApi.GetContextAsync(ConfigScope.None, cancellationToken).AsUniTask().Forget());
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(1);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenContextRequested_AndServerApiDisposed_ThenContextRequestIsCancelled()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                var systems = Create(network, out var serverApi);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenContextRequested_WithCancelledToken_ThenGetException()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    cts.Cancel();
                    await serverApi.GetAuthorizationTokenAsync(cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cts.IsCancellationRequested.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenContextRequested_AndEndpointUrlIsOffline_ThenContextShouldBeNull()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline);
                systems.InitializeAll();

                // Act
                var campaignsContext = await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);

                // Assert
                campaignsContext.Should()!.BeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(true, true)]
            [TestCase(false, false)]
            public async Task WhenContextRequested_AndCancellationTokenWasCancelled_ThenGetException(bool hasCanceled, bool expectedError)
            {
                // Arrange
                var cancellationToken = GetCancellationToken(hasCanceled ? 1 : 1000);
                var exception = default(OperationCanceledException);
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
            }

            [Test]
            [TestCase(10)]
            [TestCase(40)]
            [TestCase(200)]
            public async Task WhenGetContext_ThenContextShouldNotBeNull(int delayMs)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var contextResponse = await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);

                // Assert
                contextResponse.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(ConfigScopesCases))]
            public async Task WhenGetContextWithConfigScope_ThenContextMustContainsScopeValue(ConfigScope configScope, Func<CampaignsContext, bool> condition)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq((_, message, _) =>
                {
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var contextResponse = await serverApi.GetContextAsync(configScope, cancellationToken);

                // Assert
                contextResponse.Should()!.NotBeNull();
                condition.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(MultiplyConfigScopesCases))]
            public async Task WhenGetContextWithMultiplyConfigScope_ThenContextMustContainsScopesValue(ConfigScope configScope, Func<CampaignsContext, bool> condition)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq((_, message, _) =>
                {
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var contextResponse = await serverApi.GetContextAsync(configScope, cancellationToken);

                // Assert
                contextResponse.Should()!.NotBeNull();
                condition.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(10)]
            public async Task WhenGetContextMultiplyTimes_ThenSuccessRequestedShouldBeAsLoadingNumbers(int loadingsNumber)
            {
                // Arrange
                var successRequestedCount = 0;
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq((_, message, _) =>
                {
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                for (int i = 0; i < loadingsNumber && !serverApi.IsContextRequest; i++)
                {
                    await serverApi.GetContextAsync(ConfigScope.None, cancellationToken);
                    successRequestedCount++;
                }

                // Assert
                successRequestedCount.Should()!.Be(loadingsNumber);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(10)]
            public async Task WhenGetContextMultiplyTimes_AndFirstRequestWasCancelled_ThenSuccessRequestedShouldBeAsLoadingNumbers(int loadingsNumber)
            {
                // Arrange
                var firstCancellationToken = GetCancellationToken(1);
                var areRequestsMade = false;
                var requestCount = 0;
                var firstCancelled = false;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    // ReSharper disable once AccessToModifiedClosure
                    await UniTask.WaitUntil(() => areRequestsMade, cancellationToken: token);
                    await UniTask.Delay(10, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (areRequestsMade) requestCount++;
                    promise.TrySetResult();
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    loadingsNumber.EnumerateEach().ForEach(item =>
                    {
                        if (item == 0)
                        {
                            try
                            {
                                serverApi.GetContextAsync(ConfigScope.None, firstCancellationToken).AsUniTask().Forget();
                            }
                            catch (OperationCanceledException)
                            {
                                firstCancelled = true;
                            }
                        }
                        else
                        {
                            serverApi.GetContextAsync(ConfigScope.None, GetCancellationToken()).AsUniTask().Forget();
                        }
                    });
                    areRequestsMade = true;
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(1);
                firstCancelled.Should()!.BeTrue();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(10)]
            public async Task WhenGetContextMultiplyTimesFromThreads_ThenRequestToNetworkShouldBeOne(int loadingsNumber)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var shouldCountRequest = false;
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (shouldCountRequest) requestCount++;
                    promise.TrySetResult();
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    shouldCountRequest = true;
                    for (int i = 0; i < loadingsNumber; i++)
                    {
                        UniTask.RunOnThreadPool(() =>
                        {
                            serverApi.GetContextAsync(ConfigScope.None, cancellationToken).AsUniTask().Forget();
                        }, cancellationToken: cancellationToken).Forget();
                    }
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(1);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}