using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class AuthorizeUser : ServerApiTestsPart
        {
            private static object[] DefaultTestCases =>
                new object[]
                {
                    new object[] { RuntimePlatform.Android },
                    new object[] { RuntimePlatform.IPhonePlayer },
                };

            private static object[] MultiplyLoadingsCases =>
                new object[]
                {
                    new object[] { 2, RuntimePlatform.Android },
                    new object[] { 2, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.Android },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                };

            private static object[] CancellationTokenCancelledCases =>
                new object[]
                {
                    new object[] { true, true, RuntimePlatform.Android },
                    new object[] { true, true, RuntimePlatform.IPhonePlayer },
                    new object[] { false, false, RuntimePlatform.Android },
                    new object[] { false, false, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RequestWithDelayCases =>
                new object[]
                {
                    new object[] { 2, RuntimePlatform.Android },
                    new object[] { 2, RuntimePlatform.IPhonePlayer },
                    new object[] { 10, RuntimePlatform.Android },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                    new object[] { 40, RuntimePlatform.IPhonePlayer },
                };

            private static object[] RetryAfterExpiredErrorCases =>
                new object[]
                {
                    new object[] { 3, RuntimePlatform.Android },
                    new object[] { 3, RuntimePlatform.IPhonePlayer },
                    new object[] { 5, RuntimePlatform.Android },
                    new object[] { 5, RuntimePlatform.IPhonePlayer },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                    new object[] { 10, RuntimePlatform.IPhonePlayer },
                };

            [Test]
            [TestCaseSource(nameof(RetryAfterExpiredErrorCases))]
            public async Task WhenAuthorizeUser_AndGetExpiredErrorCode_ThenRequestShouldBeRetry(int expectedFailed, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var retries = 0;
                var network = new NetworkMoq((_, message, _) =>
                {
                    if (retries < expectedFailed)
                    {
                        retries++;
                        return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new() { Code = ErrorCode.Expired } });
                    }

                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                using var _1 = new TemporaryIgnoreFailingLogMessagesScope();
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    await serverApi.AuthorizeUser(string.Empty,  cancellationToken);
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                retries.Should()!.Be(expectedFailed);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenAuthorizeUser_AndCancelAllAuthorizeUserRequests_ThenGetOperationCancelledException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var moqReached = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    if (message.Method == ApplicationStateApiMethods.Authorize(runtimePlatform))
                    {
                        moqReached = true;
                        await UniTask.WaitWhile(() => true, cancellationToken: token);
                    }
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.WaitUntil(() => moqReached,  cancellationToken: cancellationToken);
                        serverApi.CancelAllAuthorizeUserRequests();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.AuthorizeUser(string.Empty, cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(MultiplyLoadingsCases))]
            public async Task WhenMultiplyLoadingAuthorizeUser_ThenRequestToNetworkShouldBeOne(int loadingsNumber, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var requestCount = 0;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    // ReSharper disable once AccessToModifiedClosure
                    if (message.Method == ApplicationStateApiMethods.Authorize(runtimePlatform))
                    {
                        requestCount++;
                        await UniTask.Delay(50, cancellationToken: token);
                    }
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    var tasks = loadingsNumber.EnumerateEach().Select(_ => serverApi.AuthorizeUser(string.Empty, cancellationToken));
                    await UniTask.WhenAll(tasks);
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }


                // Assert
                requestCount.Should()!.Be(1);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenServerApiDisposed_ThenAuthorizeUserIsCancelled(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var moqReached = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    moqReached = true;
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.WaitUntil(() => moqReached, cancellationToken: cancellationToken);
                        await UniTask.Yield(cancellationToken);
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.AuthorizeUser(string.Empty,  cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenAuthorizeUser_WithCancelledToken_ThenGetException(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    cts.Cancel();
                    await serverApi.AuthorizeUser(string.Empty, cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cts.IsCancellationRequested.Should()!.BeTrue();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(DefaultTestCases))]
            public async Task WhenAuthorizeUser_AndEndpointUrlIsOffline_ThenAuthorizeUserResultShouldBeNull(RuntimePlatform runtimePlatform)
            {
                // Arrange
                var network = new NetworkMoq((_, message, _) =>
                {
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                var cancellationToken = GetCancellationToken();
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var saveProgressResult = await serverApi.AuthorizeUser(string.Empty, cancellationToken);

                // Assert
                saveProgressResult.Should()!.BeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(CancellationTokenCancelledCases))]
            public async Task WhenAuthorizeUser_AndCancellationTokenWasCancelled_ThenGetException(bool mustBeCanceled, bool expectedError, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var exception = default(OperationCanceledException);
                var moqReached = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    moqReached = true;
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                try
                {
                    if (mustBeCanceled)
                    {
                        UniTask.Create(async () =>
                        {
                            await UniTask.WaitUntil(() => moqReached, cancellationToken: cts.Token);
                            cts.Cancel();
                        }).Forget();
                    }
                    await serverApi.AuthorizeUser(string.Empty, cts.Token);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                if (expectedError)
                {
                    exception.Should()!.NotBeNull();
                }
                else
                {
                    exception.Should()!.BeNull();
                }
                cts.IsCancellationRequested.Should()!.Be(mustBeCanceled);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCaseSource(nameof(RequestWithDelayCases))]
            public async Task WhenAuthorizeUserWithDelay_ThenResultShouldNotBeNull(int delayMs, RuntimePlatform runtimePlatform)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(delayMs, cancellationToken: token);
                    return NetworkMoq.OkResult(message, runtimePlatform);
                });
                using var systems = Create(network, out var serverApi, platformAPI : runtimePlatform);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                var authorizeUserResult = await serverApi.AuthorizeUser(string.Empty, cancellationToken);

                // Assert
                authorizeUserResult.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}