using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using Magify.Model;
using FluentAssertions;
using Magify.Features;
using Magify.Rx;
using NUnit.Framework;
using NSubstitute;
using UnityEngine;
using EventType = Magify.Model.EventType;

namespace Magify.Tests
{
    [TestOf(typeof(FeaturesProvider))]
    internal class FeaturesProviderTests
    {
        private class AnalyticsTrackerStub : IAnalyticsTracker
        {
            public IObservable<EventType> OnEventSent { get; }
            public IObservable<(EventType Type, IAnalyticsEvent Event)> OnEventSentDetailed { get; }
            public void HandeWentForeground() {}
            public void HandeWentBackground() {}
            public void TrackAppLaunch() {}
            public void TrackAppBackgroundEvent(long openTime, long closeTime, string appVersion) {}
            public void TrackUsedApplicationDefaultFeaturesEvent([NotNull] string featureName) {}
        }

        private const string TestConfig = "JsonModelParse/test_config";
        private const string KeyBool = "key_bool";
        private const string KeyNumber = "key_double";
        private const string KeyString = "key_string";

        private static object[] UsedApplicationDefaultFeatures_TestCaseSources
        {
            get
            {
                var list = new List<List<object>>
                {
                    new() { 3, string.Join(',', new[] { ContextKind.Default }) },
                    new() { 0, string.Join(',', new[] { ContextKind.Default, ContextKind.Saved }) },
                    new() { 0, string.Join(',', new[] { ContextKind.Default, ContextKind.Downloaded }) },
                    new() { 0, string.Join(',', new[] { ContextKind.Default, ContextKind.Saved, ContextKind.Downloaded }) },
                };
                return
                    list.Select(l => new List<object>(l.Append(true)).ToArray())
            .Concat(list.Select(l => new List<object>(l.Append(false)).ToArray()))
                    .Cast<object>()
                    .ToArray();
            }
        }

        [NotNull]
        private GeneralPrefs _prefs;
        [NotNull]
        private readonly IAnalyticsTracker _analyticsTracker = new AnalyticsTrackerStub();
        [NotNull]
        private StoredAppFeaturesProvider _storedAppFeatures;

        [SetUp]
        public void Setup()
        {
            EditorModeTestsEnvironment.Clear();
            _prefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider());
            _storedAppFeatures = new StoredAppFeaturesProvider(_prefs);
        }

        [TearDown]
        public void Cleanup()
        {
            ((IDisposable)_prefs)!.Dispose();
            ((IDisposable)_storedAppFeatures)!.Dispose();
            EditorModeTestsEnvironment.Clear();
        }

        [JetBrains.Annotations.NotNull]
        private CampaignsContext CreateContext((string, object) testFeature)
        {
            var context = new CampaignsContext();
            context.Features = new();
            context.Features.Add(testFeature.Item1, new()
            {
                DefaultValue = testFeature.Item2,
            });
            return context;
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetBool))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetNumber))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetString))]
        public void GetFeatures_AndProviderDontHaveFeatures_ThenShouldReturnEmptyDictionary()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);

            // Act
            ((IInitializable)updater).Initialize();

            // Assert
            provider.TryGetBool(KeyBool, out _).Should()!.BeFalse();
            provider.TryGetNumber(KeyNumber, out _).Should()!.BeFalse();
            provider.TryGetString(KeyString, out _).Should()!.BeFalse();
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.UpdateContext))]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void SetupFeatures_ThenShouldInvokeOnUpdateCallback()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);

            // Act
            using var updateListener = Observable.FromEvent<ConfigKind>(
                    handler => provider.OnUpdate += handler,
                    handler => provider.OnUpdate -= handler)
                .Subscribe(_ => updateCallback());

            provider.UpdateContext(context!, ContextKind.Downloaded);

            // Assert
            Assert.Fail();

            void updateCallback()
            {
                Assert.Pass();
            }
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.UpdateContext))]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void SetupDefaultFeatures_ThenShouldInvokeOnUpdateCallback()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);

            // Act
            using var updateListener = Observable.FromEvent<ConfigKind>(
                    handler => provider.OnUpdate += handler,
                    handler => provider.OnUpdate -= handler)
                .Subscribe(_ => updateCallback());

            provider.UpdateContext(context!, ContextKind.Default);

            // Assert
            Assert.Fail();

            void updateCallback()
            {
                Assert.Pass();
            }
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetBool))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetNumber))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetString))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetBool))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetNumber))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetString))]
        public void GetFeatures_AndFeaturesWasSetup_ThenShouldReturnCorrectFeatures()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);
            provider.UpdateContext(context!, ContextKind.Downloaded);

            // Act
            // Assert
            provider.GetBool(KeyBool).Value.Should()!.BeFalse();
            provider.GetNumber(KeyNumber).Value.Should()!.Be(1.2);
            provider.GetString(KeyString).Value.Should()!.Be("TESTcurrent conf 123-aaaaa");
            provider.TryGetBool(KeyBool, out var boolResult).Should()!.BeTrue();
            provider.TryGetNumber(KeyNumber, out var numberResult).Should()!.BeTrue();
            provider.TryGetString(KeyString, out var stringResult).Should()!.BeTrue();
            boolResult.Value.Should()!.Be(false);
            numberResult.Value.Should()!.Be(1.2);
            stringResult.Value.Should()!.Be("TESTcurrent conf 123-aaaaa");
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetBool))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetNumber))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.GetString))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetBool))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetNumber))]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.TryGetString))]
        public void GetDefaultFeatures_AndFeaturesWasSetup_ThenShouldReturnCorrectFeatures()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);
            provider.UpdateContext(context!, ContextKind.Default);

            // Act
            // Assert
            provider.GetBool(KeyBool).Value.Should()!.BeFalse();
            provider.GetNumber(KeyNumber).Value.Should()!.Be(1.2);
            provider.GetString(KeyString).Value.Should()!.Be("TESTcurrent conf 123-aaaaa");
            provider.TryGetBool(KeyBool, out var boolResult).Should()!.BeTrue();
            provider.TryGetNumber(KeyNumber, out var numberResult).Should()!.BeTrue();
            provider.TryGetString(KeyString, out var stringResult).Should()!.BeTrue();
            boolResult.Value.Should()!.Be(false);
            numberResult.Value.Should()!.Be(1.2);
            stringResult.Value.Should()!.Be("TESTcurrent conf 123-aaaaa");
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.UpdateContext))]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void SetupFeatures_AndSameFeaturesWasSetup_ThenShouldNotInvokeOnUpdateCallback()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);
            provider.UpdateContext(context!, ContextKind.Downloaded);

            // Act
            using var updateListener = Observable.FromEvent<ConfigKind>(
                    handler => provider.OnUpdate += handler,
                    handler => provider.OnUpdate -= handler)
                .Subscribe(_ => updateCallback());

            provider.UpdateContext(context, ContextKind.Downloaded);

            // Assert
            Assert.Pass();

            void updateCallback()
            {
                Assert.Fail();
            }
        }

        [Test]
        [UnitTestTarget(typeof(FeaturesProvider), nameof(FeaturesProvider.UpdateContext))]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void SetupDefaultFeatures_AndSameFeaturesWasSetup_ThenShouldNotInvokeOnUpdateCallback()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);
            provider.UpdateContext(context!, ContextKind.Default);

            // Act
            using var updateListener = Observable.FromEvent<ConfigKind>(
                    handler => provider.OnUpdate += handler,
                    handler => provider.OnUpdate -= handler)
                .Subscribe(_ => updateCallback());

            provider.UpdateContext(context, ContextKind.Default);

            // Assert
            Assert.Pass();

            void updateCallback()
            {
                Assert.Fail();
            }
        }

        [Test]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void ChangeInAppStatus_AndSomeFeaturesWasRemovedForNewUserStatus_ThenShouldInvokeOnUpdateCallback()
        {
            // Arrange
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            using var updater = new FeaturesUpdater(provider, _storedAppFeatures, _prefs);
            ((IInitializable)updater).Initialize();
            var json = Resources.Load<TextAsset>(TestConfig)!.text;
            var context = CampaignsContext.Deserialize(json);
            provider.UpdateContext(context!, ContextKind.Downloaded);

            // Act
            using var updateListener = Observable.FromEvent<ConfigKind>(
                    handler => provider.OnUpdate += handler,
                    handler => provider.OnUpdate -= handler)
                .Subscribe(_ => updateCallback());

            _prefs.SubscriptionStatus.Value = SubscriptionStatus.Paid;

            // Assert
            Assert.Fail();

            void updateCallback()
            {
                Assert.Pass();
            }
        }

        [Test]
        [TestCase(FeatureSource.Default, ContextKind.Default)]
        [TestCase(FeatureSource.Current, ContextKind.Saved)]
        [TestCase(FeatureSource.Current | FeatureSource.Default, ContextKind.Saved)]
        public void WhenIgnoreFeature_AndTryGetIgnoredFeature_ThenFeatureResultShouldBeFalse(FeatureSource featureSource, ContextKind contextKind)
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var testFeature = ("test_feature", "test_feature_value");
            var context = CreateContext(testFeature);

            //Act
            provider.UpdateContext(context, contextKind);
            provider.SetIgnoredFeatures(ignoreFeatures, featureSource);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            featureResult.Should()!.BeFalse();
            feature.Should()!.Be(default);
        }

        [Test]
        [TestCase(FeatureSource.Default, ContextKind.Default)]
        [TestCase(FeatureSource.Current, ContextKind.Saved)]
        [TestCase(FeatureSource.Current | FeatureSource.Default, ContextKind.Saved)]
        public void WhenClearIgnoreFeatures_AndTryGetFeature_ThenFeatureResultShouldBeTrue(FeatureSource featureSource, ContextKind contextKind)
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var testFeature = ("test_feature", "test_feature_value");
            var context = CreateContext(testFeature);

            //Act
            provider.UpdateContext(context, contextKind);
            provider.SetIgnoredFeatures(ignoreFeatures, featureSource);
            provider.SetIgnoredFeatures(null, featureSource);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            featureResult.Should()!.BeTrue();
            feature.Value.Should()!.Be("test_feature_value");
        }

        [Test]
        public void WhenIgnoreFeaturesWithCurrentFeatureSource_AndTryGetFeature_ThenFeatureShouldBeDefault()
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var defaultFeature = ("test_feature", "test_default_feature_value");
            var savedFeature = ("test_feature", "test_saved_feature_value");
            var savedContext = CreateContext(savedFeature);
            var defaultContext = CreateContext(defaultFeature);

            //Act
            provider.UpdateContext(savedContext, ContextKind.Saved);
            provider.UpdateContext(defaultContext, ContextKind.Default);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Current);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            featureResult.Should()!.BeTrue();
            feature.Value.Should()!.Be("test_default_feature_value");
        }

        [Test]
        public void WhenIgnoreFeaturesWithAllFeatureSource_AndTryGetFeature_ThenFeatureShouldBeFalse()
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var defaultFeature = ("test_feature", "test_default_feature_value");
            var savedFeature = ("test_feature", "test_saved_feature_value");
            var savedContext = CreateContext(savedFeature);
            var defaultContext = CreateContext(defaultFeature);

            //Act
            provider.UpdateContext(savedContext, ContextKind.Saved);
            provider.UpdateContext(defaultContext, ContextKind.Default);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Current);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Default);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            featureResult.Should()!.BeFalse();
            feature.Should()!.Be(default);
        }

        [Test]
        public void WhenIgnoreFeaturesWithAllFeatureSource_AndGetFeature_ThenExceptionShouldBeNotNull()
        {
            //Arrange
            var exception = default(MagifyFeatureNotFoundException);
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var feature = ("test_feature", "test_default_feature_value");
            var context = CreateContext(feature);

            //Act
            provider.UpdateContext(context, ContextKind.Saved);
            provider.UpdateContext(context, ContextKind.Default);
            provider.UpdateContext(context, ContextKind.Downloaded);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Default | FeatureSource.Current);
            try
            {
                provider.GetString("test_feature");
            }
            catch (MagifyFeatureNotFoundException e)
            {
                exception = e;
            }

            //Assert
            exception.Should()!.NotBeNull();
            exception.FeatureName.Should()!.Be("test_feature");
            exception.FeatureType.Should()!.Be(typeof(string));
        }

        [Test]
        public void WhenIgnoreCachedFeature_AndGetFeatureAfterDownloadContext_ThenFeatureShouldBeDefault()
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var savedFeature = ("test_feature", "test_saved_feature_value");
            var defaultFeature = ("test_feature", "test_default_feature_value");
            var downloadedFeature = ("test_feature", "test_downloaded_feature_value");
            var savedContext = CreateContext(savedFeature);
            var defaultContext = CreateContext(defaultFeature);
            var downloadedContext = CreateContext(downloadedFeature);

            //Act
            provider.UpdateContext(savedContext, ContextKind.Saved);
            provider.UpdateContext(defaultContext, ContextKind.Default);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Current);
            provider.UpdateContext(downloadedContext, ContextKind.Downloaded);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            provider.GetString("test_feature");

            //Assert
            featureResult.Should()!.BeTrue();
            feature.Value.Should()!.Be(defaultFeature.Item2);
        }

        [Test]
        public void WhenClearIgnoreFeatures_AndUpdateDownloadedContext_ThenFeatureShouldBeDownloadedValue()
        {
            //Arrange
            var ignoreFeatures = new List<string>(){"test_feature"};
            using var provider = new FeaturesProvider(_prefs, _analyticsTracker);
            var savedFeature = ("test_feature", "test_saved_feature_value");
            var defaultFeature = ("test_feature", "test_default_feature_value");
            var downloadedFeature = ("test_feature", "test_downloaded_feature_value");
            var savedContext = CreateContext(savedFeature);
            var defaultContext = CreateContext(defaultFeature);
            var downloadedContext = CreateContext(downloadedFeature);

            //Act
            provider.UpdateContext(savedContext, ContextKind.Saved);
            provider.UpdateContext(defaultContext, ContextKind.Default);
            provider.SetIgnoredFeatures(ignoreFeatures, FeatureSource.Current);
            provider.SetIgnoredFeatures(null, FeatureSource.Current);
            provider.UpdateContext(downloadedContext, ContextKind.Downloaded);
            var featureResult = provider.TryGetString("test_feature", out var feature);

            //Assert
            provider.GetString("test_feature");

            //Assert
            featureResult.Should()!.BeTrue();
            feature.Value.Should()!.Be(downloadedFeature.Item2);
        }

        [TestCaseSource(nameof(UsedApplicationDefaultFeatures_TestCaseSources))]
        public void UpdateContext_GetFeature_ThenUsedApplicationDefaultFeaturesEventTrackedSomeTimes(int eventsNumber, string contextKinds, bool tryGet)
        {
            // Arrange
            using var subsystems = CreateFeaturesProvider(out var provider, out var analyticsTracker);
            var json = Resources.Load<TextAsset>(TestConfig).text;
            var context = CampaignsContext.Deserialize(json);
            subsystems.ForEach<IInitializable>(s => s.Initialize());
            contextKinds.Split(',')
                .Select(Enum.Parse<ContextKind>)
                .ForEach(kind => subsystems.ForEach<IContextListener>(s => s.UpdateContext(context, kind)));
            var eventsTracked = 0;
            using var disposable = analyticsTracker.OnEventSent
                .Where(e => e is EventType.UsedApplicationDefaultFeatures)
                .Subscribe(_ => eventsTracked++);

            // Act
            if (tryGet)
            {
                provider.TryGetBool(KeyBool, out _);
                provider.TryGetNumber(KeyNumber, out _);
                provider.TryGetString(KeyString, out _);
            }
            else
            {
                provider.GetBool(KeyBool);
                provider.GetNumber(KeyNumber);
                provider.GetString(KeyString);
            }

            // Assert
            eventsTracked.Should().Be(eventsNumber);
        }

        [Test]
        [TestCaseSource(nameof(UsedApplicationDefaultFeatures_TestCaseSources))]
        public void UpdateContext_GetUndefinedFeature_ThenUsedApplicationDefaultFeaturesEventNotTracked(int __, string contextKinds, bool tryGet)
        {
            // Arrange
            using var subsystems = CreateFeaturesProvider(out var provider, out var analyticsTracker);
            var json = Resources.Load<TextAsset>(TestConfig).text;
            var context = CampaignsContext.Deserialize(json);
            subsystems.ForEach<IInitializable>(s => s.Initialize());
            contextKinds.Split(',')
                .Select(Enum.Parse<ContextKind>)
                .ForEach(kind => subsystems.ForEach<IContextListener>(s => s.UpdateContext(context, kind)));
            var eventsTracked = 0;
            using var disposable = analyticsTracker.OnEventSent
                .Where(e => e is EventType.UsedApplicationDefaultFeatures)
                .Subscribe(_ => eventsTracked++);

            // Act
            const string undefinedKey = "undefined_key_413586130!@#$%%&(_)";
            if (tryGet)
            {
                provider.TryGetBool(undefinedKey, out _);
                provider.TryGetNumber(undefinedKey, out _);
                provider.TryGetString(undefinedKey, out _);
            }
            else
            {
                try
                {
                    provider.GetBool(undefinedKey);
                    provider.GetNumber(undefinedKey);
                    provider.GetString(undefinedKey);
                }
                catch (MagifyFeatureNotFoundException)
                {
                    // It's ok
                }
            }

            // Assert
            eventsTracked.Should().Be(0);
        }

        private SubsystemsCollection CreateFeaturesProvider(out FeaturesProvider featuresProvider) => CreateFeaturesProvider(out featuresProvider, out _);

        private SubsystemsCollection CreateFeaturesProvider(out FeaturesProvider featuresProvider, out AnalyticsTracker analyticsTracker)
        {
            var subsystems = new SubsystemsCollection();
            var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
            var mockServerApi = Substitute.For<IServerApi>();
            var appStatePrefs = AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath).AddTo(subsystems);
            var platform = new PlatformDefault().AddTo(subsystems);
            analyticsTracker = new AnalyticsTracker(mockServerApi, _prefs, platform, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(subsystems);
            featuresProvider = new FeaturesProvider(_prefs, analyticsTracker).AddTo(subsystems);
            return subsystems;
        }
    }
}