using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Magify.Types;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using NSubstitute;
using NUnit.Framework;
using UnityEditor;
using UnityEngine.Networking;

namespace Magify.Tests
{
    internal class AnalyticsTrackerTests
    {
        public delegate Action<object> CreateEventRequestHandlerDelegate(ReactiveProperty<bool?> isEventBasic);

        [NotNull]
        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        private PlatformAPI _platformAPI;
        [NotNull]
        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        private NetworkClient _realNetworkClient;
        private WebResponseMessage _interceptedResponseMessage;

        [NotNull]
        // ReSharper disable once NotNullOrRequiredMemberIsNotInitialized
        private CancellationTokenSource _timeoutCts;


        private static object[] IsBasicIntegrationCases =>
            new object[]
            {
                new object[] {
                    EventType.AppLaunch,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var appLaunchEvent = @event as AppLaunchEvent;
                            isEventBasic.Value = appLaunchEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackAppLaunch },
                new object[] {
                    EventType.AppBackgroundEvent,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var appBackgroundEvent = @event as AppBackgroundEvent;
                            isEventBasic.Value = appBackgroundEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackBackgroundPurchaseEvent },
                new object[] {
                    EventType.ApplovinAdsImpression,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var adsImpressionEvent = @event as AdsImpressionEvent;
                            isEventBasic.Value = adsImpressionEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackAdsImpression },
                new object[] {
                    EventType.InApp,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var productPurchaseEvent = @event as ProductPurchaseEvent;
                            isEventBasic.Value = productPurchaseEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackInAppPurchaseEvent },
                new object[] {
                    EventType.FirebaseMapping,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var firebaseMappingEvent = @event as FirebaseMappingEvent;
                            isEventBasic.Value = firebaseMappingEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackFirebaseMappingEvent },
                new object[] {
                    EventType.MailingStatusMapping,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var mailingStatusMappingEvent = @event as MailingStatusMappingEvent;
                            isEventBasic.Value = mailingStatusMappingEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackMailingStatusMappingEvent },
                new object[] {
                    EventType.ImpressionFail,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var impressionFailEvent = @event as ImpressionFailEvent;
                            isEventBasic.Value = impressionFailEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackImpressionFailEvent },
                new object[] {
                    EventType.Impression,
                    (CreateEventRequestHandlerDelegate)((ReactiveProperty<bool?> isEventBasic) => {
                        return (Action<object>)(@event => {
                            var campaignImpressionEvent = @event as CampaignImpressionEvent;
                            isEventBasic.Value = campaignImpressionEvent?.IsBasicIntegration;
                        });
                    }),
                    (Action<AnalyticsTracker>)AnalyticsTrackerUtils.TrackCampaignImpressionEvent },
            };

        [SetUp]
        public void Setup()
        {
            EditorModeTestsEnvironment.Clear();
            _platformAPI = new PlatformDefault();
            _realNetworkClient = new NetworkClient();
            _interceptedResponseMessage = default;
            _timeoutCts = new CancellationTokenSource(TimeSpan.FromSeconds(3));
        }

        [TearDown]
        public void Cleanup()
        {
            _platformAPI.Dispose();
            _timeoutCts.Cancel();
            _timeoutCts.Dispose();
            EditorModeTestsEnvironment.Clear();
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackCustomSessionEvent))]
        public async Task TrackCustomSessionEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Act
            analyticsTracker.TrackCustomSessionEvent("custom_session_event", new Dictionary<string, object>());
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackTransaction))]
        public async Task TrackIncomeTransaction_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Act
            analyticsTracker.TrackTransaction(
                Transaction.CreateIncome("source"),
                new List<BonusInfo>
                {
                    new()
                    {
                        BonusName = "bonus"
                    },
                },
                new ProductInfo("product", "1", "USD", PurchaseStore.Native),
                new CampaignImpression
                {
                    CampaignName = "test",
                    EventName = "test",
                    EventNumber = 1,
                    Parameters = new Dictionary<string, object>()
                });
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackFirebaseMappingEvent))]
        public async Task TrackFirebaseMappingEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Act
            analyticsTracker.TrackFirebaseMappingEvent("123456");
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackMailingStatusMappingEvent))]
        public async Task TrackMailingStatusMappingEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Act
            analyticsTracker.TrackMailingStatusMappingEvent("test@gmail.com");
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackApplovinAdsImpression))]
        public async Task TrackApplovinImpressionEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            var impression = new CampaignImpression
            {
                CampaignName = "test",
                EventName = "test",
                EventNumber = 1,
                Parameters = new Dictionary<string, object>()
            };
            var applovinData = new ApplovinAdsImpression
            {
                Id = "testId",
                Network = "testNetwork",
                ImpressionId = GUID.Generate().ToString(),
                Revenue = 0.1,
                Precision = "test"
            };

            // Act
            analyticsTracker.TrackApplovinAdsImpression(impression, applovinData);
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackIronSourceAdsImpression))]
        public async Task TrackIronSourceImpressionEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            var impression = new CampaignImpression
            {
                CampaignName = "test",
                EventName = "test",
                EventNumber = 1
            };
            var applovinData = new IronSourceAdsImpression
            {
                Id = "testId",
                Network = "testNetwork",
                AdUnit = "banner",
                Revenue = 0.1,
                Precision = "test"
            };

            // Act
            analyticsTracker.TrackIronSourceAdsImpression(impression, applovinData);
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackPurchaseEvent))]
        public async Task TrackSubscriptionPurchaseEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            var impression = new CampaignImpression
            {
                CampaignName = "test",
                EventName = "test",
                EventNumber = 1
            };
            var product = new ProductInfo("testId", "9.99", "USD", PurchaseStore.Native);
            var purchase = new PurchaseInfo
            {
                Product = product
            };

            // Act
            analyticsTracker.TrackPurchaseEvent(EventType.TrialActivation, impression, purchase);
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackCommonEvent))]
        public async Task TrackCampaignImpression_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            var impression = new CampaignImpression
            {
                CampaignName = "testCampaign",
                EventName = "testEvent",
                Parameters = new Dictionary<string, object>(),
                ImpressionNumber = 1,
                SessionImpressionNumber = 1,
                CampaignTypeImpressionNumber = 1,
                CampaignTypeSessionImpressionNumber = 1,
                EventNumber = 1,
                SessionEventNumber = 1,
            };

            // Act
            analyticsTracker.TrackCommonEvent(EventType.Impression, impression);
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackImpressionFailEvent))]
        public async Task TrackImpressionFailEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            var impression = new CampaignImpression
            {
                CampaignName = "testCampaign",
                EventName = "testEvent",
                Parameters = new Dictionary<string, object>(),
                ImpressionNumber = 1,
                SessionImpressionNumber = 1,
                CampaignTypeImpressionNumber = 1,
                CampaignTypeSessionImpressionNumber = 1,
                EventNumber = 1,
                SessionEventNumber = 1,
            };

            // Act
            analyticsTracker.TrackImpressionFailEvent(impression, "test");
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackAppLaunch))]
        public async Task TrackAppLaunch_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out _);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Act
            analyticsTracker.TrackAppLaunch();
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        [UnitTestTarget(typeof(AnalyticsTracker), nameof(AnalyticsTracker.TrackAppBackgroundEvent))]
        public async Task TrackSessionEndEvent_ThenShouldSucceedSendEventToServer()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out var networkClient, out var prefs);
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            prefs.SessionId.Value = Guid.NewGuid().ToString();

            // Act
            analyticsTracker.TrackAppBackgroundEvent(DateTimeOffset.UtcNow.ToUnixTimeMilliseconds() - 3000, DateTimeOffset.UtcNow.ToUnixTimeMilliseconds(), new AppVersionProvider().AppVersion);
            await TaskHelper.WaitUntil(() => _interceptedResponseMessage.Result != UnityWebRequest.Result.InProgress, _timeoutCts.Token);

            // Assert
            FailIfTimeoutTokenWasCanceled();
            networkClient
                .Received(2)!
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>()).Forget();
            _interceptedResponseMessage.ResponseCode.Should()!.Be(200);
            _interceptedResponseMessage.Result.Should()!.Be(UnityWebRequest.Result.Success);
        }

        [Test]
        public void WhenAnalyticsTrackerStarted_ThanAppLaunchEventSent()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out _, out var prefs, out var appStatePrefs);

            var platform = new PlatformDefault().AddTo(subsystems);
            new AnalyticsSessionTracker(analyticsTracker, prefs, appStatePrefs, platform, new AppVersionProvider()).AddTo(subsystems);
            new SessionCounter(prefs, platform).AddTo(subsystems);

            var appLaunchEvents = 0;
            using var eventObserver = analyticsTracker
                .OnEventSent
                .Where(e => e is EventType.AppLaunch)
                .Subscribe(_ => appLaunchEvents++);

            // Act
            subsystems.ForEach<IInitializable>(s => s!.Initialize());

            // Assert
            appLaunchEvents.Should()!.Be(1);
        }

        [Test]
        public void WhenNewSessionStartedAfterForeground_ThanAppLaunchEventSent()
        {
            // Arrange
            using var subsystems = CreateAnalyticsTracker(out var analyticsTracker, out _, out var prefs, out var appStatePrefs);

            var platform = new PlatformDefault().AddTo(subsystems);
            new AnalyticsSessionTracker(analyticsTracker, prefs, appStatePrefs, platform, new AppVersionProvider()).AddTo(subsystems);
            var sessionCounter = new SessionCounter(prefs, platform).AddTo(subsystems);
            sessionCounter.Interval.Value = TimeSpan.Zero;

            var appLaunchEvents = 0;
            using var eventObserver = analyticsTracker
                .OnEventSent
                .Where(e => e is EventType.AppLaunch)
                .Subscribe(_ => appLaunchEvents++);

            // Act
            subsystems.ForEach<IInitializable>(s => s!.Initialize());
            subsystems.ForEach<IBackgroundListener>(s => s!.OnBackground());
            subsystems.ForEach<IForegroundListener>(s => s!.OnForeground());

            // Assert
            appLaunchEvents.Should()!.Be(2);
        }

        [Test]
        public void WhenAppQuitedWithoutGoingToBackground_AndStartedAgain_ThanAppBackgroundEventSent()
        {
            // Arrange
            using (var subsystems = create(out _))
            {
                subsystems.ForEach<IInitializable>(i => i!.Initialize());
            }
            var sessionEndEvents = 0;

            // Act
            using (var subsystems = create(out var analyticsTracker))
            {
                using var eventObserver = analyticsTracker
                    .OnEventSent
                    .Where(e => e is EventType.AppBackgroundEvent)
                    .Subscribe(_ => sessionEndEvents++);

                subsystems.ForEach<IInitializable>(i => i!.Initialize());
            }

            // Assert
            sessionEndEvents.Should()!.Be(1);

            [NotNull]
            SubsystemsCollection create([NotNull] out AnalyticsTracker analyticsTracker)
            {
                var subsystems = CreateAnalyticsTracker(out analyticsTracker, out _, out var prefs, out var appStatePrefs);
                var platform = new PlatformDefault().AddTo(subsystems);
                new SessionCounter(prefs, platform).AddTo(subsystems);
                new AnalyticsSessionTracker(analyticsTracker, prefs, appStatePrefs, platform, new AppVersionProvider()).AddTo(subsystems);
                return subsystems;
            }
        }

        [Test]
        [TestCaseSource(nameof(IsBasicIntegrationCases))]
        public async Task WhenTrackEvent_ThenIsBasicPropertyShouldBeNull(EventType eventType, [NotNull] CreateEventRequestHandlerDelegate createEventRequestHandler, Action<AnalyticsTracker> trackAction)
        {
            //Arrange
            var isEventBasic = new ReactiveProperty<bool?>(null);
            var handleEventRequest = createEventRequestHandler(isEventBasic);
            using var systems = AnalyticsTrackerUtils.Create(eventType, handleEventRequest, out AnalyticsTracker analyticsTracker, out var serverApi);
            analyticsTracker.SetupAnalyticsConfig(new AnalyticsConfiguration());
            systems.InitializeAll();
            await serverApi.CancelAllServerInteractions();

            //Act
            trackAction?.Invoke(analyticsTracker);

            //Assert
            isEventBasic.Value.Should()!.BeNull();
        }

        [NotNull]
        private INetworkClient CreateNetworkClient()
        {
            var networkClient = Substitute.For<INetworkClient>();

            // ReSharper disable once PossibleNullReferenceException
            networkClient
                .SendAsync(Arg.Any<string>(), Arg.Any<WebRequestMessage>(), Arg.Any<CancellationToken>())
                .Returns<UniTask<WebResponseMessage>>(
                    async info => await _realNetworkClient.SendAsync(info!.ArgAt<string>(0)!, info.ArgAt<WebRequestMessage>(1), info.ArgAt<CancellationToken>(2)),
                    async info => _interceptedResponseMessage = await _realNetworkClient.SendAsync(info!.ArgAt<string>(0)!, info.ArgAt<WebRequestMessage>(1), info.ArgAt<CancellationToken>(2)));
            return networkClient;
        }

        [NotNull]
        private GeneralPrefs CreateGeneralPrefs()
        {
            return GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, new AppVersionProvider());
        }

        [NotNull]
        private AppStatePrefs CreateAppStatePrefs()
        {
            return AppStatePrefs.Create(EditorModeTestsEnvironment.AppStatePrefsPath);
        }

        private void FailIfTimeoutTokenWasCanceled()
        {
            if (_timeoutCts.IsCancellationRequested)
            {
                Assert.Fail("Timeout token was canceled");
            }
        }

        [NotNull]
        private SubsystemsCollection CreateAnalyticsTracker([NotNull] out AnalyticsTracker analyticsTracker, [NotNull] out INetworkClient networkClient, [NotNull] out GeneralPrefs prefs)
            => CreateAnalyticsTracker(out analyticsTracker, out networkClient, out prefs, out var _);

        [NotNull]
        private SubsystemsCollection CreateAnalyticsTracker([NotNull] out AnalyticsTracker analyticsTracker, [NotNull] out INetworkClient networkClient, [NotNull] out GeneralPrefs prefs, [NotNull] out AppStatePrefs appStatePrefs)
        {
            var subsystems = new SubsystemsCollection();
            var platform = new PlatformDefault().AddTo(subsystems);
            var appVersionProvider = new AppVersionProvider().AddTo(subsystems);
            prefs = CreateGeneralPrefs().AddTo(subsystems);
            appStatePrefs = CreateAppStatePrefs().AddTo(subsystems);
            networkClient = CreateNetworkClient().AddTo(subsystems);
            var mockServerApi = Substitute.For<IServerApi>();
            analyticsTracker = new AnalyticsTracker(mockServerApi!, prefs, platform, appStatePrefs, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(subsystems);
            return subsystems;
        }
    }
}