using System;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class SingleModeRequestToolsTests
    {
        [Test]
        public void WhenCreateNewCompletionSource_ThenCompletionTaskShouldNotBeNull()
        {
            //Arrange
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken,out UniTask<object>? completionTask, out CancellationToken singleCancellationToken);

            //Assert
            completionTask.Should()!.NotBeNull();
            completionTask.Value.Status.Should()!.Be(UniTaskStatus.Pending);
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeFalse();
        }

        [Test]
        public void WhenNotCreateNewCompletionSource_ThenIsThereActiveCompletionTaskShouldBeFalse()
        {
            //Arrange
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act

            //Assert
            singleModRequestTools.IsThereActiveCompletionTask().Should()!.BeFalse();
        }

        [Test]
        public void WhenCreateNewCompletionSource_ThenIsThereActiveCompletionTaskShouldBeTrue()
        {
            //Arrange
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken, out _, out CancellationToken singleCancellationToken);

            //Assert
            singleModRequestTools.IsThereActiveCompletionTask().Should()!.BeTrue();
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeFalse();
        }

        [Test]
        public void WhenCreateNewCompletionSource_AndDispose_ThenIsThereActiveCompletionTaskShouldBeFalse()
        {
            //Arrange
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken, out _, out CancellationToken singleCancellationToken);
            singleModRequestTools.Dispose();

            //Assert
            singleModRequestTools.IsThereActiveCompletionTask().Should()!.BeFalse();
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeTrue();
        }

        [Test]
        public async Task WhenCreateNewCompletionSource_AndTrySetResult_ThenCompletionTaskShouldNotBeNull()
        {
            //Arrange
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken, out UniTask<object>? completionTask, out CancellationToken singleCancellationToken);
            singleModRequestTools.TrySetResult(new());
            var result = default(object);
            try
            {
                result = await completionTask.Value;
            }
            catch (OperationCanceledException)
            {
                // ignore
            }

            //Assert
            completionTask.Should()!.NotBeNull();
            result.Should()!.NotBeNull();
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeTrue();
        }

        [Test]
        public async Task WhenCreateNewCompletionSource_AndTrySetCancelled_ThenCompletionTaskShouldBeNull()
        {
            //Arrange
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken, out UniTask<object>? completionTask, out CancellationToken singleCancellationToken);
            singleModRequestTools.TrySetCancelled(new());
            var result = default(object);
            try
            {
                result = await completionTask.Value;
            }
            catch (OperationCanceledException)
            {
                // ignore
            }

            //Assert
            completionTask.Should()!.NotBeNull();
            completionTask.Value.Status.Should()!.Be(UniTaskStatus.Canceled);
            result.Should()!.BeNull();
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeTrue();
        }

        [Test]
        public void WhenCreateNewCompletionSource_AndTrySetException_ThenExceptionShouldNotBeNull()
        {
            //Arrange
            var exception = new Exception();
            var cancellationToken = new CancellationToken();
            var singleModRequestTools = new SingleModeRequestTools<object>();

            //Act
            CancellationToken singleCancellationToken;
            try
            {
                singleModRequestTools.TryCreateNewCompletionSource(in cancellationToken, out _, out singleCancellationToken);
                singleModRequestTools.TrySetException(exception);
            }
            catch (Exception e)
            {
                exception.Should()!.BeSameAs(e);
                Assert.Pass();
            }

            //Assert
            exception.Should()!.NotBeNull();
            cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            singleCancellationToken.IsCancellationRequested.Should()!.BeTrue();
        }
    }
}