﻿using System;
using System.Collections.Generic;
using System.Linq;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using Random = UnityEngine.Random;

namespace Magify.Tests
{
    internal class CampaignModelExtensionsTest
    {
        private const string TestBundleId = "com.test.app";
        private const string TestScheme = "89QWERTYUIOP";

        private static int[] RandomNums => Enumerable.Range(0, 5).Select(_ => (int)Random.value).ToArray();

        [Test]
        [TestCaseSource(nameof(IsCampaignRelevant_TestCases))]
        public void IsCampaignRelevant(CampaignModel campaign, Func<PlatformAPI> platformGetter, bool isRelevant)
        {
            // Arrange
            using var platform = platformGetter();

            // Act
            var result = campaign.IsCampaignRelevant(platform);

            // Assert
            result.Should().Be(isRelevant);
        }

        [Test]
        [TestCaseSource(nameof(TryGetNested_TestCases))]
        public void TryGetNested(CampaignModel campaign, string[] productIds, NestedCampaign[] expectedNested)
        {
            // Act
            var results = productIds
                .Select(p => (ThereIs: campaign.TryGetNested(p, out var result), Result: result, Product: p))
                .Select(r => r.ThereIs && r.Result == expectedNested[Array.IndexOf(productIds, r.Product)]);

            // Assert
            results.Should().AllBeEquivalentTo(true);
        }

        private static IEnumerable<object[]> IsCampaignRelevant_TestCases()
        {
            // Non relevant Android
            yield return new object[]
            {
                campaign(),
                (Func<PlatformAPI>)(() =>
                {
                    var platform = new PlatformDefault(EditorDevice.GetDefaultAndroidDevice());
                    platform.TrackPromotedApplication(TestBundleId);
                    return platform;
                }),
                false,
            };

            // Non relevant iOS
            yield return new object[]
            {
                campaign(),
                (Func<PlatformAPI>)(() =>
                {
                    var platform = new PlatformDefault(EditorDevice.GetDefaultIOSDevice());
                    platform.TrackPromotedApplication(TestScheme);
                    return platform;
                }),
                false,
            };

            // Relevant Android
            yield return new object[]
            {
                campaign(),
                (Func<PlatformAPI>)(() => new PlatformDefault(EditorDevice.GetDefaultAndroidDevice())),
                true,
            };

            // Relevant iOS
            yield return new object[]
            {
                campaign(),
                (Func<PlatformAPI>)(() => new PlatformDefault(EditorDevice.GetDefaultIOSDevice())),
                true,
            };

            CampaignModel campaign()
            {
                return new CampaignModel
                {
                    Type = CampaignType.CrossPromo,
                    Info = new MetaInfo
                    {
                        Application = new PromotedApplication
                        {
                            BundleId = TestBundleId,
                            Schemas = new List<string> {TestScheme},
                        },
                    },
                };
            }
        }

        private static IEnumerable<object[]> TryGetNested_TestCases()
        {
            var oneNested = RandomNums.Select(i => $"product{i}")
                .Select(createItem)
                .Select(item => new object[] { item.CampaignModel, item.ProductIds, item.Nesteds, });
            var manyNesteds = RandomNums.Select(i => new[] { $"product{i}", $"product{i + 1}" })
                .Select(createItems)
                .Select(item => new object[] { item.CampaignModel, item.ProductIds, item.Nesteds, });
            return oneNested.Concat(manyNesteds);

            (CampaignModel CampaignModel, string[] ProductIds, NestedCampaign[] Nesteds) createItem(string productId)
            {
                var nested = new NestedCampaign { ProductId = productId, };
                var campaignModel = new CampaignModel { NestedCampaigns = new List<NestedCampaign> { nested }, };
                return (campaignModel, new[] { productId }, new[] { nested });
            }

            (CampaignModel CampaignModel, string[] ProductIds, NestedCampaign[] Nesteds) createItems(string[] productIds)
            {
                var nesteds = productIds.Select(p => new NestedCampaign { ProductId = p }).ToArray();
                var campaignModel = new CampaignModel { NestedCampaigns = nesteds.ToList(), };
                return (campaignModel, productIds, nesteds);
            }
        }
    }
}