using System;
using System.Threading;
using System.Timers;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class SyncTimer : ICancelable
    {
        private readonly int _minSyncInterval;
        [NotNull]
        private readonly System.Timers.Timer _timer = new();
        [NotNull]
        private readonly Action _onElapsedAction;
        [NotNull]
        private readonly CancellationTokenSource _cts = new();

        public bool IsDisposed { get; private set; }

        public SyncTimer([NotNull] Action onElapsedAction, int minSyncInterval = 1000)
        {
            if (minSyncInterval <= 0)
            {
                throw new ArgumentOutOfRangeException($"{nameof(minSyncInterval)} cannot be equal to or less than 0");
            }
            _onElapsedAction = onElapsedAction;
            _minSyncInterval = minSyncInterval;
        }

        public void Start(int syncInterval)
        {
            ThrowIfDisposed();

            _timer.Interval = Math.Max(_minSyncInterval, syncInterval);
            _timer.Elapsed += OnElapsed;
            _timer.Start();
        }

        private void OnElapsed(object sender, ElapsedEventArgs args) => OnElapsedAsync().Forget();
        private async UniTaskVoid OnElapsedAsync()
        {
            try
            {
                if (IsDisposed)
                    return;
                await TaskScheduler.SwitchToMainThread(_cts.Token);
                if (IsDisposed)
                    return;
            }
            catch (OperationCanceledException) { /* ignore */ }
            _onElapsedAction();
            _timer.Start();
        }

        public void Restart(int? syncInterval = null)
        {
            ThrowIfDisposed();

            _timer.Stop();
            if (syncInterval.HasValue)
            {
                _timer.Interval = Math.Max(_minSyncInterval, syncInterval.Value);
            }
            _timer.Start();
        }

        public void Stop()
        {
            ThrowIfDisposed();
            _timer.Stop();
        }

        public void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            _cts.Cancel();
            _cts.Dispose();
            _timer.Elapsed -= OnElapsed;
            _timer.Stop();
            IsDisposed = true;
        }

        private void ThrowIfDisposed()
        {
            if (IsDisposed)
            {
                throw new MagifyObjectDisposedException(nameof(SyncTimer));
            }
        }
    }
}