using System;
using JetBrains.Annotations;
using Newtonsoft.Json;
using UnityEngine.Scripting;

namespace Magify
{
    [Preserve]
    public class UnitTimeSpanConverter : JsonConverter
    {
        public TimeUnit TimeUnit { get; set; }
        public bool SupportFloatingNumbers { get; set; }

        [UsedImplicitly, Preserve]
        public UnitTimeSpanConverter() : this(TimeUnit.Ticks, false)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitTimeSpanConverter(TimeUnit timeUnit) : this(timeUnit, false)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitTimeSpanConverter(bool supportFloatingNumbers) : this(TimeUnit.Ticks, supportFloatingNumbers)
        {
        }

        [UsedImplicitly, Preserve]
        public UnitTimeSpanConverter(TimeUnit timeUnit, bool supportFloatingNumbers)
        {
            TimeUnit = timeUnit;
            SupportFloatingNumbers = supportFloatingNumbers;
        }

        public override bool CanConvert(Type objectType)
        {
            return objectType == typeof(TimeSpan) || objectType == typeof(TimeSpan?);
        }

        public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
        {
            if (value is TimeSpan span)
            {
                writer.WriteValue(span, TimeUnit, SupportFloatingNumbers);
            }
            else
            {
                writer.WriteNull();
            }
        }

        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            var timeSpan = reader.ReadTimeSpan(TimeUnit, SupportFloatingNumbers);
            if (timeSpan == null)
            {
                var nullable = objectType.IsGenericType && objectType.GetGenericTypeDefinition() == typeof(Nullable<>);
                return nullable ? null : throw new JsonSerializationException($"Cannot convert {reader.Value} value to {objectType.Name}.");
            }
            return timeSpan.Value;
        }
    }
}