using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using UnityEngine.Pool;

namespace Magify
{
    public static class DictionaryExtensions
    {
        public static bool Compare<TKey, TValue>(this Dictionary<TKey, TValue> x, Dictionary<TKey, TValue> y)
        {
            if (x.Count != y.Count) return false;
            if (x.Keys.Except(y.Keys).Any()) return false;
            if (y.Keys.Except(x.Keys).Any()) return false;
            return x.All(pair => Equals(pair.Value, y[pair.Key]));
        }

        /// <summary>
        /// Replaces the int representation of the CampaignType type with a string representation
        /// </summary>
        /// <returns>Number of replaced values</returns>
        public static int ReplaceCampaignTypeWithString([NotNull] this IDictionary<string, object> dictionary)
        {
            if (dictionary.Values.All(isNotCampaignType))
            {
                return 0;
            }

            var keys = ListPool<string>.Get()!;
            keys.Clear();
            keys.AddRange(dictionary.Keys);

            var replaced = 0;
            foreach (var key in keys)
            {
                var value = dictionary[key];
                if (value is CampaignType campaignType)
                {
                    dictionary[key] = campaignType.ToEnumString();
                    replaced++;
                }
            }

            keys.Clear();
            ListPool<string>.Release(keys);
            return replaced;

            bool isNotCampaignType<T>([CanBeNull] T value) => value is not CampaignType;
        }

        public static void Copy<TKey, TValue>([NotNull] this IDictionary<TKey, TValue> dictionary, [CanBeNull] IReadOnlyDictionary<TKey, TValue> source)
        {
            if (source == null || source.Count == 0)
            {
                return;
            }

            foreach (var item in source)
            {
                if (item.Key != null)
                {
                    dictionary[item.Key] = item.Value;
                }
            }
        }
    }
}