using System;
using System.Collections;
using System.Collections.Generic;
using JetBrains.Annotations;

namespace Magify
{
    internal class ReactiveHashSet<T> : IReactiveCollection, IRefillableCollection<T>, ISet<T>, IReadOnlyCollection<T>
    {
        [NotNull]
        private readonly HashSet<T> _set = new();

        public event Action OnChanged;

        private void SetDirty()
        {
            OnChanged?.Invoke();
        }

        #region Mutable functionallity

        public void CopyTo(T[] array, int arrayIndex)
        {
            _set.CopyTo(array, arrayIndex);
        }

        void ICollection<T>.Add(T item)
        {
            if (_set.Add(item))
            {
                SetDirty();
            }
        }

        public void ExceptWith(IEnumerable<T> other)
        {
            var count = Count;
            _set.ExceptWith(other);
            if (Count != count)
            {
                SetDirty();
            }
        }

        public void IntersectWith(IEnumerable<T> other)
        {
            var count = Count;
            _set.IntersectWith(other);
            if (Count != count)
            {
                SetDirty();
            }
        }

        public void SymmetricExceptWith(IEnumerable<T> other)
        {
            var count = Count;
            _set.SymmetricExceptWith(other);
            if (Count != count)
            {
                SetDirty();
            }
        }

        public void UnionWith(IEnumerable<T> other)
        {
            var count = Count;
            _set.UnionWith(other);
            if (Count != count)
            {
                SetDirty();
            }
        }

        public bool Add(T item)
        {
            var added = _set.Add(item);
            if (added)
            {
                SetDirty();
            }
            return added;
        }

        public void Clear()
        {
            var count = Count;
            _set.Clear();
            if (Count != count)
            {
                SetDirty();
            }
        }

        public bool Remove(T item)
        {
            var removed = _set.Remove(item);
            if (removed)
            {
                SetDirty();
            }
            return removed;
        }

        public bool Contains(T item)
        {
            return _set.Contains(item);
        }

        public void Refill(IEnumerable<T> elements)
        {
            var changed = _set.Count;
            _set.Clear();
            changed += _set.AddRange(elements);
            if (changed > 0)
                SetDirty();
        }

        #endregion

        #region Immutable functionallity

        public int Count => _set.Count;

        public bool IsReadOnly => ((ISet<T>)_set).IsReadOnly;

        public IEnumerator<T> GetEnumerator() => _set.GetEnumerator();

        IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();

        public bool IsProperSubsetOf(IEnumerable<T> other) => _set.IsProperSubsetOf(other);

        public bool IsProperSupersetOf(IEnumerable<T> other) => _set.IsProperSupersetOf(other);

        public bool IsSubsetOf(IEnumerable<T> other) => _set.IsSubsetOf(other);

        public bool IsSupersetOf(IEnumerable<T> other) => _set.IsSupersetOf(other);

        public bool Overlaps(IEnumerable<T> other) => _set.Overlaps(other);

        public bool SetEquals(IEnumerable<T> other) => _set.SetEquals(other);

        #endregion
    }
}