﻿using System;
using System.Buffers;
using System.Collections.Generic;
using System.IO;
using System.Text;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    public abstract class BinaryTypeHandler<T>
    {
        [NotNull]
        public virtual string TypeName { get; } = typeof(T).Name;
        public abstract int SizeOf(T value);
        public abstract void WriteTo(BinaryWriter writer, T value);
        public abstract T ReadFrom(BinaryReader reader);
    }

    public class BinaryEnumHandler<TEnum, TNumber> : BinaryTypeHandler<TEnum>
        where TEnum : unmanaged, Enum
        where TNumber : unmanaged
    {
        [NotNull]
        private readonly BinaryTypeHandler<TNumber> _numberType;

        public override string TypeName { get; }
        public override int SizeOf(TEnum value) => _numberType.SizeOf(ToNumber(value));

        public BinaryEnumHandler([NotNull] BinaryTypeHandler<TNumber> numberType, bool useFullName)
        {
            _numberType = numberType;
            TypeName = (useFullName ? typeof(TEnum).FullName : typeof(TEnum).Name)!;
        }

        public override void WriteTo(BinaryWriter writer, TEnum value)
        {
            var number = ToNumber(value);
            _numberType.WriteTo(writer, number);
        }

        public override TEnum ReadFrom(BinaryReader reader)
        {
            var number = _numberType.ReadFrom(reader);
            return FromNumber(number);
        }

        public static unsafe TNumber ToNumber(TEnum enumValue)
        {
            return *(TNumber*)(&enumValue);
        }

        public static unsafe TEnum FromNumber(TNumber numberValue)
        {
            return *(TEnum*)(&numberValue);
        }
    }

    internal class BinaryTypesPair<TType1, TType2> : BinaryTypeHandler<KeyValuePair<TType1, TType2>>
    {
        [NotNull]
        private readonly BinaryTypeHandler<TType1> _type1;
        [NotNull]
        private readonly BinaryTypeHandler<TType2> _type2;

        public override string TypeName { get; }

        public BinaryTypesPair([NotNull] BinaryTypeHandler<TType1> t1, [NotNull] BinaryTypeHandler<TType2> t2)
        {
            _type1 = t1;
            _type2 = t2;
            TypeName = $"{typeof(TType1).Name}:{typeof(TType2).Name}";
        }

        public override int SizeOf(KeyValuePair<TType1, TType2> value)
        {
            return _type1.SizeOf(value.Key) + _type2.SizeOf(value.Value);
        }

        public override void WriteTo(BinaryWriter writer, KeyValuePair<TType1, TType2> value)
        {
            _type1.WriteTo(writer, value.Key);
            _type2.WriteTo(writer, value.Value);
        }

        public override KeyValuePair<TType1, TType2> ReadFrom(BinaryReader reader)
        {
            return new KeyValuePair<TType1, TType2>(_type1.ReadFrom(reader), _type2.ReadFrom(reader));
        }
    }

    internal class BinaryTypeBoolean : BinaryTypeHandler<bool>
    {
        [NotNull]
        public static BinaryTypeBoolean Shared { get; } = new();
        public override int SizeOf(bool _) => sizeof(bool);
        public override void WriteTo([NotNull] BinaryWriter writer, bool value) => writer.Write(value);
        public override bool ReadFrom([NotNull] BinaryReader reader) => reader.ReadBoolean();
    }

    internal class BinaryTypeChar : BinaryTypeHandler<char>
    {
        [NotNull]
        public static BinaryTypeChar Shared { get; } = new();
        public override int SizeOf(char _) => sizeof(char);
        public override void WriteTo([NotNull] BinaryWriter writer, char value) => writer.Write(value);
        public override char ReadFrom([NotNull] BinaryReader reader) => reader.ReadChar();
    }

    internal class BinaryTypeByte : BinaryTypeHandler<byte>
    {
        [NotNull]
        public static BinaryTypeByte Shared { get; } = new();
        public override int SizeOf(byte _) => sizeof(byte);
        public override void WriteTo([NotNull] BinaryWriter writer, byte value) => writer.Write(value);
        public override byte ReadFrom([NotNull] BinaryReader reader) => reader.ReadByte();
    }

    internal class BinaryTypeSByte : BinaryTypeHandler<sbyte>
    {
        [NotNull]
        public static BinaryTypeSByte Shared { get; } = new();
        public override int SizeOf(sbyte _) => sizeof(sbyte);
        public override void WriteTo([NotNull] BinaryWriter writer, sbyte value) => writer.Write(value);
        public override sbyte ReadFrom([NotNull] BinaryReader reader) => reader.ReadSByte();
    }

    internal class BinaryTypeInt16 : BinaryTypeHandler<short>
    {
        [NotNull]
        public static BinaryTypeInt16 Shared { get; } = new();
        public override int SizeOf(short _) => sizeof(short);
        public override void WriteTo([NotNull] BinaryWriter writer, short value) => writer.Write(value);
        public override short ReadFrom([NotNull] BinaryReader reader) => reader.ReadInt16();
    }

    internal class BinaryTypeUInt16 : BinaryTypeHandler<ushort>
    {
        [NotNull]
        public static BinaryTypeUInt16 Shared { get; } = new();
        public override int SizeOf(ushort _) => sizeof(short);
        public override void WriteTo([NotNull] BinaryWriter writer, ushort value) => writer.Write(value);
        public override ushort ReadFrom([NotNull] BinaryReader reader) => reader.ReadUInt16();
    }

    internal class BinaryTypeInt32 : BinaryTypeHandler<int>
    {
        [NotNull]
        public static BinaryTypeInt32 Shared { get; } = new();
        public override int SizeOf(int _) => sizeof(int);
        public override void WriteTo([NotNull] BinaryWriter writer, int value) => writer.Write(value);
        public override int ReadFrom([NotNull] BinaryReader reader) => reader.ReadInt32();
    }

    internal class BinaryTypeUInt32 : BinaryTypeHandler<uint>
    {
        [NotNull]
        public static BinaryTypeUInt32 Shared { get; } = new();
        public override int SizeOf(uint _) => sizeof(uint);
        public override void WriteTo([NotNull] BinaryWriter writer, uint value) => writer.Write(value);
        public override uint ReadFrom([NotNull] BinaryReader reader) => reader.ReadUInt32();
    }

    internal class BinaryTypeInt64 : BinaryTypeHandler<long>
    {
        [NotNull]
        public static BinaryTypeInt64 Shared { get; } = new();
        public override int SizeOf(long _) => sizeof(long);
        public override void WriteTo([NotNull] BinaryWriter writer, long value) => writer.Write(value);
        public override long ReadFrom([NotNull] BinaryReader reader) => reader.ReadInt64();
    }

    internal class BinaryTypeUInt64 : BinaryTypeHandler<ulong>
    {
        [NotNull]
        public static BinaryTypeUInt64 Shared { get; } = new();
        public override int SizeOf(ulong _) => sizeof(ulong);
        public override void WriteTo([NotNull] BinaryWriter writer, ulong value) => writer.Write(value);
        public override ulong ReadFrom([NotNull] BinaryReader reader) => reader.ReadUInt64();
    }

    internal class BinaryTypeSingle : BinaryTypeHandler<float>
    {
        [NotNull]
        public static BinaryTypeSingle Shared { get; } = new();
        public override int SizeOf(float _) => sizeof(float);
        public override void WriteTo([NotNull] BinaryWriter writer, float value) => writer.Write(value);
        public override float ReadFrom([NotNull] BinaryReader reader) => reader.ReadSingle();
    }

    internal class BinaryTypeDouble : BinaryTypeHandler<double>
    {
        [NotNull]
        public static BinaryTypeDouble Shared { get; } = new();
        public override int SizeOf(double _) => sizeof(double);
        public override void WriteTo([NotNull] BinaryWriter writer, double value) => writer.Write(value);
        public override double ReadFrom([NotNull] BinaryReader reader) => reader.ReadDouble();
    }

    internal class BinaryTypeDecimal : BinaryTypeHandler<decimal>
    {
        [NotNull]
        public static BinaryTypeDecimal Shared { get; } = new();
        public override int SizeOf(decimal _) => sizeof(decimal);
        public override void WriteTo([NotNull] BinaryWriter writer, decimal value) => writer.Write(value);
        public override decimal ReadFrom([NotNull] BinaryReader reader) => reader.ReadDecimal();
    }

    internal class BinaryTypeString : BinaryTypeHandler<string>
    {
        [NotNull]
        private readonly Encoding _encoding;

        [NotNull]
        public static BinaryTypeString Shared { get; } = new(Encoding.UTF8);

        public BinaryTypeString() : this(Encoding.UTF8)
        {
        }

        public BinaryTypeString([NotNull] Encoding encoding)
        {
            _encoding = encoding;
        }

        public override int SizeOf(string value) => sizeof(int) + (value != null ? _encoding.GetByteCount(value) : 0);

        public override void WriteTo([NotNull] BinaryWriter writer, string value)
        {
            if (value == null)
            {
                writer.Write(-1);
            }
            else if (value.Length == 0)
            {
                writer.Write(0);
            }
            else
            {
                var size = SizeOf(value);
                var buffer = ArrayPool<byte>.Shared?.Rent(size) ?? new byte[size];
                var bufferSize = _encoding.GetBytes(value, buffer);
                writer.Write(bufferSize);
                writer.Write(buffer, 0, bufferSize);
                ArrayPool<byte>.Shared?.Return(buffer);
            }
        }

        public override string ReadFrom([NotNull] BinaryReader reader)
        {
            var size = reader.ReadInt32();
            if (size == -1)
            {
                return null;
            }
            if (size == 0)
            {
                return string.Empty;
            }
            var buffer = ArrayPool<byte>.Shared?.Rent(size) ?? new byte[size];
            var read = reader.BaseStream.Read(buffer, 0, size);
            if (read != size)
            {
                ArrayPool<byte>.Shared?.Return(buffer);
                return string.Empty;
            }
            var value = _encoding.GetString(buffer, 0, size);
            ArrayPool<byte>.Shared?.Return(buffer);
            return value;
        }
    }

    internal class BinaryTypeDateTime : BinaryTypeHandler<DateTime>
    {
        [NotNull]
        public static BinaryTypeDateTime Shared { get; } = new();

        public override int SizeOf(DateTime _) => sizeof(long);

        public override void WriteTo([NotNull] BinaryWriter writer, DateTime value)
        {
            writer.Write(value.ToBinary());
        }

        public override DateTime ReadFrom([NotNull] BinaryReader reader)
        {
            var ticks = reader.ReadInt64();
            return DateTime.FromBinary(ticks);
        }
    }

    internal class BinaryTypeDateTimeLegacy : BinaryTypeHandler<DateTime>
    {
        [NotNull]
        public static BinaryTypeDateTimeLegacy Shared { get; } = new();

        public override int SizeOf(DateTime _) => sizeof(long);

        public override void WriteTo([NotNull] BinaryWriter writer, DateTime value)
        {
            var unixMilliseconds = (value - DateTime.UnixEpoch).TotalMilliseconds;
            writer.Write(unixMilliseconds);
        }

        public override DateTime ReadFrom([NotNull] BinaryReader reader)
        {
            var unixMilliseconds = reader.ReadDouble();
            return DateTime.UnixEpoch.AddMilliseconds(unixMilliseconds);
        }
    }

    internal class BinaryTypeVector2 : BinaryTypeHandler<Vector2>
    {
        [NotNull]
        public static BinaryTypeVector2 Shared { get; } = new();
        public override int SizeOf(Vector2 _) => sizeof(float) * 2;

        public override void WriteTo([NotNull] BinaryWriter writer, Vector2 value)
        {
            writer.Write(value.x);
            writer.Write(value.y);
        }

        public override Vector2 ReadFrom([NotNull] BinaryReader reader)
        {
            var x = reader.ReadSingle();
            var y = reader.ReadSingle();
            return new Vector2(x, y);
        }
    }

    internal class BinaryTypeVector3 : BinaryTypeHandler<Vector3>
    {
        [NotNull]
        public static BinaryTypeVector3 Shared { get; } = new();
        public override int SizeOf(Vector3 _) => sizeof(float) * 3;

        public override void WriteTo([NotNull] BinaryWriter writer, Vector3 value)
        {
            writer.Write(value.x);
            writer.Write(value.y);
            writer.Write(value.z);
        }

        public override Vector3 ReadFrom([NotNull] BinaryReader reader)
        {
            var x = reader.ReadSingle();
            var y = reader.ReadSingle();
            var z = reader.ReadSingle();
            return new Vector3(x, y, z);
        }
    }

    internal class BinaryTypeNullable<T> : BinaryTypeHandler<T?>
        where T: struct
    {
        [NotNull]
        public static BinaryTypeNullable<T> Shared { get; } = new();

        public override int SizeOf(T? value)
        {
            return
                sizeof(bool) // there is value or not
              + BinaryTypeHandlerUtils.HandlerForPrimitive<T>().SizeOf(value ?? default); // value size
        }

        public override void WriteTo([NotNull] BinaryWriter writer, T? value)
        {
            var handler = BinaryTypeHandlerUtils.HandlerForPrimitive<T>();
            writer.Write(value.HasValue);
            handler.WriteTo(writer, value ?? default);
        }

        public override T? ReadFrom([NotNull] BinaryReader reader)
        {
            var handler = BinaryTypeHandlerUtils.HandlerForPrimitive<T>();
            var hasValue = reader.ReadBoolean();
            var value = handler.ReadFrom(reader);
            return hasValue ? value : null;
        }
    }
}