using System;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class SessionCounter : IInitializable, IForegroundListener, IBackgroundListener, IDisposable
    {
        public static readonly TimeSpan DefaultSessionInterval = TimeSpan.FromSeconds(2);

        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        private readonly GeneralPrefs _prefs;
        private readonly PlatformAPI _platformAPI;

        private DateTime _shiftToBackgroundTime = DateTime.MaxValue;

        [NotNull]
        public IReactiveProperty<TimeSpan> Interval { get; set; } = new ReactiveProperty<TimeSpan>(DefaultSessionInterval);

        public SessionCounter(GeneralPrefs prefs, PlatformAPI platformAPI)
        {
            _prefs = prefs;
            _platformAPI = platformAPI;
        }

        void IInitializable.Initialize()
        {
            StartNewSession();
        }

        public void ForceStartNewSession()
        {
            StartNewSession();
        }

        private void StartNewSession()
        {
            _prefs.SessionId.Value = Guid.NewGuid().ToString();
            _prefs.VersionSessionCounter++;
            _prefs.GlobalSessionCounter.Value++;
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();
        }

        void IForegroundListener.OnForeground()
        {
            if ((DateTime.UtcNow - _shiftToBackgroundTime) > Interval.Value)
            {
                StartNewSession();
            }
        }

        void IBackgroundListener.OnBackground()
        {
            _shiftToBackgroundTime = DateTime.UtcNow;
        }
    }
}