using System;
using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class PurchaseTracker : ILegacyMigrator, IInitializable, IForegroundListener, IBackgroundListener, ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.Purchases);

        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly PurchasesStorage _storage;
        [NotNull]
        private readonly List<PurchaseRecord> _uploadingPurchaseRecords = new();
        [NotNull]
        private readonly List<TrustedPurchaseRecord> _uploadingTrustedPurchaseRecords = new();
        [NotNull]
        private readonly SyncTimer _syncTimer;
        [NotNull]
        private readonly EmbeddedPurchaseVerificationHandler _embeddedPurchaseVerificationHandler = new();
        [NotNull, ItemNotNull]
        private List<PurchaseRecord> _unsentPurchaseRecords = new();
        [NotNull, ItemNotNull]
        private List<TrustedPurchaseRecord> _unsentTrustedPurchaseRecords = new();
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        public bool IsDisposed => _disposables.IsDisposed;

        [NotNull]
        public ReactiveProperty<IPurchaseVerificationHandler> ExternalPurchaseVerificationHandler { get; } = new(null);
        [NotNull]
        public ReactiveProperty<float> VerificationRetryInterval { get; } = new(60.0f);
        private int RetryIntervalMills => Mathf.Max(Mathf.RoundToInt(VerificationRetryInterval.Value * 1000), 1000);

        public PurchaseTracker([NotNull] IServerApi serverApi, PlatformAPI platform, string storagePath)
        {
            _serverApi = serverApi;
            _platform = platform;
            _storage = new PurchasesStorage(storagePath);
            _syncTimer = new SyncTimer(() =>
            {
                SendRecords(_unsentPurchaseRecords, SendPurchaseRecord);
                SendRecords(_unsentTrustedPurchaseRecords, SendTrustedPurchaseRecord);
            }).AddTo(_disposables)!;
            ExternalPurchaseVerificationHandler
                .Subscribe(handler => _logger.Log($"External purchase verification handler changed to {handler?.GetType().Name ?? "null"}"))
                .AddTo(_disposables);
            VerificationRetryInterval
                .Subscribe(_ =>
                {
                    _logger.Log($"Verification retry interval changed to {VerificationRetryInterval.Value} seconds");
                    _syncTimer?.Restart(RetryIntervalMills);
                })
                .AddTo(_disposables);
        }

        public void Initialize()
        {
            ThrowIfDisposed();
            _unsentPurchaseRecords = _storage.LoadPurchaseRecords();
            _unsentTrustedPurchaseRecords = _storage.LoadTrustedPurchaseRecords();
            _syncTimer.Start(RetryIntervalMills);
            SendPurchaseRecords();
            SendTrustedPurchaseRecords();
        }

        public void ValidateInApp([NotNull] string productId, string transactionId = null, string purchaseToken = null, string receipt = null)
        {
            ValidatePurchase(productId, false, transactionId, purchaseToken, receipt);
        }

        public void ValidateSubscription([NotNull] string productId, string transactionId = null, string purchaseToken = null, string receipt = null)
        {
            ValidatePurchase(productId, true, transactionId, purchaseToken, receipt);
        }

        public void ValidatePurchase([NotNull] string productId, bool isSubscription, string transactionId = null, string purchaseToken = null, string receipt = null)
        {
            ThrowIfDisposed();
            var purchaseRecord = new PurchaseRecord
            {
                ProductId = productId,
#if UNITY_ANDROID
                IsSubscription = isSubscription,
                TransactionId = transactionId,
                PurchaseToken = purchaseToken,
#elif UNITY_IOS
                Receipt = receipt,
                DeviceId = _platform.GetEncodedDeviceIdentifier(),
                BuildNumber = _platform.BuildNumber,
#endif
            };
            AddUnsentPurchaseRecord(purchaseRecord);
            SendPurchaseRecords();
            _syncTimer.Restart();
        }

        public void TrackTrustedPurchase([NotNull] TrustedPurchaseRecord record)
        {
            ThrowIfDisposed();
            AddUnsentTrustedPurchaseRecord(record);
            SendTrustedPurchaseRecords();
            _syncTimer.Restart();
        }

        private void SendPurchaseRecords()
        {
            SendRecords(_unsentPurchaseRecords, SendPurchaseRecord);
        }

        private void SendTrustedPurchaseRecords()
        {
            SendRecords(_unsentTrustedPurchaseRecords, SendTrustedPurchaseRecord);
        }

        private void SendRecords<T>([NotNull] IList<T> records, [NotNull] Func<T, CancellationToken, UniTaskVoid> sendRecordsFunc)
        {
            for (var i = records.Count - 1; i >= 0; i--)
            {
                var record = records[i];
                if (record != null)
                {
                    try
                    {
                        sendRecordsFunc(record, _disposables.GetOrCreateToken()).Forget();
                    }
                    catch (Exception e)
                    {
                        _logger.LogWarning($"Failed to send purchase record with exception {e}");
                    }
                }
            }
        }

        private async UniTaskVoid SendTrustedPurchaseRecord([NotNull] TrustedPurchaseRecord record, CancellationToken cancellationToken)
        {
            if (_uploadingTrustedPurchaseRecords.Contains(record))
            {
                return;
            }

            var retryImmediately = false;
            try
            {
                _uploadingTrustedPurchaseRecords.Add(record);
                await TaskScheduler.SwitchToThreadPool(cancellationToken);
                var result = await _serverApi.SendTrustedPurchase(record, cancellationToken);
                await TaskScheduler.SwitchToMainThread(cancellationToken);

                if (result)
                {
                    RemoveSentTrustedPurchaseRecord(record);
                }
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Verification request for {record.ProductId} was cancelled, retrying immediately");
                retryImmediately = true; // we can set `true` here because below we check `IsDisposed` and only then call `_disposables.GetOrCreateToken()`
            }
            finally
            {
                if (!IsDisposed)
                {
                    _uploadingTrustedPurchaseRecords.Remove(record);
                    if (retryImmediately)
                    {
                        _logger.Log($"Immediately retrying to send trusted purchase {record.ProductId}");
                        SendTrustedPurchaseRecord(record, _disposables.GetOrCreateToken()).Forget();
                    }
                    else
                    {
                        _syncTimer.Restart();
                    }
                }
            }
        }

        private async UniTaskVoid SendPurchaseRecord([NotNull] PurchaseRecord record, CancellationToken cancellationToken)
        {
            if (_uploadingPurchaseRecords.Contains(record))
            {
                return;
            }

            var retryImmediately = false;
            try
            {
                _uploadingPurchaseRecords.Add(record);
                await TaskScheduler.SwitchToThreadPool(cancellationToken);
                var result = await _serverApi.VerifyPurchase(record, waitForResult: ExternalPurchaseVerificationHandler.Value != null, cancellationToken);
                await TaskScheduler.SwitchToMainThread(cancellationToken);

                var handlerResult = ExternalPurchaseVerificationHandler.Value?.HandlePurchaseVerification(result)
                                 ?? _embeddedPurchaseVerificationHandler.HandlePurchaseVerification(result);
                _logger.Log($"For purchase {record.ProductId} received verification result: {result.Code}, handler result: {handlerResult}");
                if (handlerResult is RepeatState.Finish ||
                    result.Code is PurchaseVerificationResultCode.Success or PurchaseVerificationResultCode.Invalid)
                {
                    RemoveSentPurchaseRecord(record);
                }
                else if (handlerResult is RepeatState.Retry)
                {
                    retryImmediately = true;
                }
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Verification request for {record.ProductId} was cancelled, retrying immediately");
                retryImmediately = true; // we can set `true` here because below we check `IsDisposed` and only then call `_disposables.GetOrCreateToken()`
            }
            finally
            {
                if (!IsDisposed)
                {
                    _uploadingPurchaseRecords.Remove(record);
                    if (retryImmediately)
                    {
                        _logger.Log($"Immediately retrying verification of purchase {record.ProductId}");
                        SendPurchaseRecord(record, _disposables.GetOrCreateToken()).Forget();
                    }
                    else
                    {
                        _syncTimer.Restart();
                    }
                }
            }
        }

        private void AddUnsentTrustedPurchaseRecord([NotNull] TrustedPurchaseRecord record)
        {
            AddUnsentRecord(record, _unsentTrustedPurchaseRecords, records => _storage.SaveTrustedPurchaseRecords(records!));
        }

        private void AddUnsentPurchaseRecord([NotNull] PurchaseRecord record)
        {
            AddUnsentRecord(record, _unsentPurchaseRecords, records => _storage.SavePurchaseRecords(records!));
        }

        private void AddUnsentRecord<T>([NotNull] T record, [NotNull, ItemNotNull] List<T> records, [NotNull] Action<List<T>> saveRecords)
        {
            _logger.Log($"Add unsent record: {JsonFacade.SerializeObject(record)}");
            records.Add(record);
            saveRecords.Invoke(records);
        }

        private void RemoveUnsentRecord<T>([NotNull] T record, [NotNull, ItemNotNull] List<T> records, [NotNull] Action<List<T>> saveRecords)
        {
            _logger.Log($"Remove sent record: {JsonFacade.SerializeObject(record)}");
            records.Remove(record);
            saveRecords.Invoke(records);
        }

        private void RemoveSentTrustedPurchaseRecord([NotNull] TrustedPurchaseRecord record)
        {
            RemoveUnsentRecord(record, _unsentTrustedPurchaseRecords, records => _storage.SaveTrustedPurchaseRecords(records!));
        }

        private void RemoveSentPurchaseRecord([NotNull] PurchaseRecord record)
        {
            RemoveUnsentRecord(record, _unsentPurchaseRecords, records => _storage.SavePurchaseRecords(records!));
        }

        void IForegroundListener.OnForeground()
        {
            _syncTimer.Start(RetryIntervalMills);
        }

        void IBackgroundListener.OnBackground()
        {
            _syncTimer.Stop();
            _storage.SavePurchaseRecords(_unsentPurchaseRecords);
            _storage.SaveTrustedPurchaseRecords(_unsentTrustedPurchaseRecords);
        }

        private void ThrowIfDisposed()
        {
            if (IsDisposed)
            {
                throw new ObjectDisposedException(nameof(PurchaseTracker));
            }
        }

        public void Dispose()
        {
            ThrowIfDisposed();
            _disposables.Release();
            _uploadingPurchaseRecords.Clear();
        }

        public void Migrate([NotNull] MigrationData data)
        {
            _logger.Log("Try to migrate purchases from native");
            if (_storage.PurchasesAreExist())
            {
                _logger.Log("Saved purchases already exists. Skip migration.");
                return;
            }
            if (data.PurchasedRecords == null)
            {
                _logger.Log("No purchases to migration data. Skip migration.");
                return;
            }
            _storage.SavePurchaseRecords(data.PurchasedRecords!);
            _logger.Log($"Purchases successfully migrated in folder: {_storage.RootFolderPath}");
        }
    }
}