#if UNITY_EDITOR || UNITY_IOS
using System;
using System.Linq;
using System.Collections.Generic;
using Magify.Model;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using Newtonsoft.Json;
using UnityEngine;

namespace Magify
{
    internal partial class PlatformIOS : PlatformAPI, IInitializable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private static readonly MagifyLogger _migrationLogger = MagifyLogger.Get(LoggingScope.Migration);
        [NotNull]
        private readonly PooledCompositeDisposable _disposable = new();
        private static PlatformIOS _callbacksSingleton;
        [NotNull]
        private readonly IOSLegacyCountersStorage _legacyCountersStorage;
        [NotNull]
        private readonly Subject<(string Key, string Value)> _onCloudStoreValueFetched = new();
        [NotNull]
        private readonly Subject<object> _onBackground = new();
        [NotNull]
        private readonly Subject<object> _onForeground = new();
        [NotNull]
        private readonly Subject<object> _onApplicationDidBecomeActive = new();
        [NotNull]
        private readonly Subject<object> _onApplicationWillResignActive = new();
        [NotNull]
        private readonly Subject<LocaleData> _onLocaleChanged = new();
        [NotNull]
        private readonly Subject<string> _onNewPackageInstalled = new();
        [CanBeNull]
        private LocaleData _mockedLocale = null;
        [CanBeNull]
        private string _cachedDeviceName;
        [CanBeNull]
        private string _cachedDeviceVersion;
        [CanBeNull]
        private string _cachedBuildNumber;
        [CanBeNull]
        private string _cachedIdfa;
        [CanBeNull]
        private string _cachedIdfv;
        [CanBeNull]
        private string _cachedStoreCountry;
        [CanBeNull]
        private string _cachedEncodedDeviceIdentifier;


        public override RuntimePlatform RuntimePlatform => RuntimePlatform.IPhonePlayer;
        /// <remarks>
        /// Important: switching to main thread may cause foreground/background events to be skipped when enter background/overlapping with native UI.
        /// </remarks>
        [NotNull]
        public override IObservable<object> OnBackground => _onBackground;
        [NotNull]
        public override IObservable<object> OnForeground => _onForeground;
        [NotNull]
        public override IObservable<object> OnApplicationDidBecomeActive => _onApplicationDidBecomeActive;
        [NotNull]
        public override IObservable<object> OnApplicationWillResignActive => _onApplicationWillResignActive;
        [NotNull]
        public override IObservable<LocaleData> OnLocaleChanged => _onLocaleChanged.ObserveOnMainThread(_disposable.GetOrCreateToken());
        [NotNull]
        public override IObservable<string> OnNewPackageInstalled => _onNewPackageInstalled.ObserveOnMainThread(_disposable.GetOrCreateToken());
        [NotNull]
        public override string DeviceName => _cachedDeviceName ?? _magifyGetDeviceModel();
        public override string DeviceVersion => _cachedDeviceVersion ?? _magifyGetOsVersion();
        public override string BuildNumber
        {
            get
            {
                return
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
                    CustomBuildNumber ??
#endif
                    _cachedBuildNumber ?? _magifyGetBuildNumber();
            }
        }

        public override bool IsTablet
        {
            get
            {
                var device = DeviceName;
                if (device.Contains("iPad"))
                {
                    return true;
                }
                if (device.Contains("iPhone"))
                {
                    return false;
                }
                return base.IsTablet;
            }
        }

        public PlatformIOS()
        {
            _callbacksSingleton = this;

            var countersBinaryStoragePath = Path.Combine(PackageInfo.PersistentPath, "iOS_legacy", "counters.bin");
            _legacyCountersStorage = new IOSLegacyCountersStorage(countersBinaryStoragePath);
            _magifyInitialize();
        }

        void IInitializable.Initialize()
        {
            _magifySetOnCloudStoreValueFetched(OnCloudStoreValueFetchedHandler);
            _magifySetOnApplicationEnterBackground(OnApplicationEnterBackgroundHandler);
            _magifySetOnApplicationEnterForeground(OnApplicationEnterForegroundHandler);
            _magifySetOnApplicationDidBecomeActive(OnApplicationDidBecomeActiveHandler);
            _magifySetOnApplicationWillResignActive(OnApplicationWillResignActiveHandler);
            _cachedDeviceName = _magifyGetDeviceModel();
            _cachedDeviceVersion = _magifyGetOsVersion();
            _cachedBuildNumber = _magifyGetBuildNumber();
            _cachedIdfa = _magifyGetIdfa();
            _cachedIdfv = _magifyGetIdfv();
            _cachedStoreCountry = _magifyGetStoreCountry();
            _cachedEncodedDeviceIdentifier = _magifyGetEncodedDeviceIdentifier();
        }

        [NotNull]
        public override LocaleData GetCurrentLocale()
        {
            return _mockedLocale ?? new LocaleData(_magifyGetLanguageWithRegion());
        }

        public override void AddPlatformSpecificData([NotNull] AuthTokenPayload authToken, [NotNull] GeneralPrefs prefs, bool isGeoIpEnabled, bool syncStateEnabled)
        {
            base.AddPlatformSpecificData(authToken, prefs, isGeoIpEnabled, syncStateEnabled);
            // Pass null if isGeoIPEnabled == true (absent parameter will be treated by server as true)
            // In order to hide this parameter for sake of anti-fraud
            authToken.IsGeoIPEnabled = isGeoIpEnabled ? null : false;
            authToken.IsIdfaAvailable = prefs.AttAuthorizationStatus.Value;
            authToken.Idfa = GetIdfa();
            authToken.Idfv = GetIdfv();
        }

        public override void TweakUserLocale(string languageTag)
        {
            _mockedLocale = new LocaleData("US", languageTag);
            _onLocaleChanged.OnNext(_mockedLocale);
        }

        public override void TrackPromotedApplication(string bundleId)
        {
            // Do nothing
        }

        public override bool IsApplicationInstalled(string urlScheme)
        {
            return _magifyIsApplicationInstalled(urlScheme);
        }

        public override bool IsApplicationInstalled(PromotedApplication application)
        {
            return application?.Schemas?.Any(_magifyIsApplicationInstalled) ?? false;
        }

        public override string GetIdfa()
        {
            return CustomIdfa ?? _cachedIdfa ?? _magifyGetIdfa();
        }

        public override string GetIdfv()
        {
            return CustomIdfv ?? _cachedIdfv ?? _magifyGetIdfv();
        }

        public override string GetEncodedDeviceIdentifier()
        {
            return _cachedEncodedDeviceIdentifier ?? _magifyGetEncodedDeviceIdentifier();
		}

        public override string GetStoreCountry()
        {
            return _cachedStoreCountry ?? _magifyGetStoreCountry();
        }

        public override string GetValueInNativeStorage(string key)
        {
            try
            {
                var value = _magifyGetStringValueInKeychain(key);
                _logger.Log($"{nameof(GetValueInNativeStorage)} called for key {key} and returned value {value}");
                return value;
            }
            catch (Exception e)
            {
                throw new MagifyFailedToAccessToNativeStorageException(
                    MagifyFailedToAccessToNativeStorageException.NativeStorageKind.IOSKeychain,
                    MagifyFailedToAccessToNativeStorageException.NativeStorageAccessKind.Get,
                    key,
                    e);
            }
        }

        public override void SetValueInNativeStorage(string value, string key)
        {
            try
            {
                _logger.Log($"{nameof(SetValueInNativeStorage)} called for key {key} with value {value}");
                _magifySetStringValueInKeychain(value, key);
            }
            catch (Exception e)
            {
                throw new MagifyFailedToAccessToNativeStorageException(
                    MagifyFailedToAccessToNativeStorageException.NativeStorageKind.IOSKeychain,
                    MagifyFailedToAccessToNativeStorageException.NativeStorageAccessKind.Set,
                    key,
                    e);
            }
        }

        public override void RemoveValueInNativeStorage(string key)
        {
            try
            {
                _logger.Log($"{nameof(RemoveValueInNativeStorage)} called for key {key}");
                _magifyRemoveValueInKeychain(key);
            }
            catch (Exception e)
            {
                throw new MagifyFailedToAccessToNativeStorageException(
                    MagifyFailedToAccessToNativeStorageException.NativeStorageKind.IOSKeychain,
                    MagifyFailedToAccessToNativeStorageException.NativeStorageAccessKind.Remove,
                    key,
                    e);
            }
        }

        public override void ClearNativeStorage()
        {
            _magifyClearKeychains();
            _logger.Log("Keychains cleared");
        }

        public override async UniTask<string> GetValueFromCloudStorage([NotNull] string key, double timeoutSeconds, CancellationToken cancellationToken)
        {
            try
            {
                var completion = new UniTaskCompletionSource<string>();
                using var _ = _onCloudStoreValueFetched
                    .Where(result => result.Key == key)
                    .Subscribe(result => completion.TrySetResult(result.Value));
                _magifyFetchValueFromCloudStore(key);
                var delayTask = UniTask.Delay(TimeSpan.FromSeconds(timeoutSeconds), cancellationToken: cancellationToken);
                var (hasResult, value) = await UniTask.WhenAny(completion.Task, delayTask);
                if (hasResult)
                    return value;
                _logger.LogWarning($"Magify failed to get value from cloud storage for key: {key} and within timeout {timeoutSeconds} seconds.");
            }
            catch (Exception e)
            {
                _logger.LogException(new MagifyFailedToGetValueFromCloudStorageException(key, e));
            }
            return default(string);
        }

        public override void SetValueInCloudStorage([NotNull] string key, [NotNull] string value)
        {
            try
            {
                _magifySetValueToCloudStore(value, key);
                _logger.Log($"Magify successfully set value to cloud storage, key: {key}, value: {value}.");
            }
            catch (Exception e)
            {
                _logger.LogException(new MagifyFailedToSetValueToCloudStorageException(key, value, e));
            }
        }

        public override void RemoveValueFromCloudStorage([NotNull] string key)
        {
            try
            {
                _magifyRemoveValueFromCloudStore(key);
                _logger.Log($"Magify successfully removed value from cloud storage, key: {key}.");
            }
            catch (Exception e)
            {
                _logger.LogException(new MagifyFailedToRemoveValueFromCloudStorageException(key, e));
            }
        }

        [AOT.MonoPInvokeCallback(typeof(OnCloudStoreValueFetchedCallback))]
        private static void OnCloudStoreValueFetchedHandler(string key, string value)
        {
            _callbacksSingleton?._onCloudStoreValueFetched.OnNext((key, value));
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationEnterBackgroundHandler()
        {
            _callbacksSingleton?._onBackground.OnNext(_callbacksSingleton);
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationEnterForegroundHandler()
        {
            _callbacksSingleton?._onForeground.OnNext(_callbacksSingleton);
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationDidBecomeActiveHandler()
        {
            _callbacksSingleton?._onApplicationDidBecomeActive.OnNext(_callbacksSingleton);
        }

        [AOT.MonoPInvokeCallback(typeof(Action))]
        private static void OnApplicationWillResignActiveHandler()
        {
            _callbacksSingleton?._onApplicationWillResignActive.OnNext(_callbacksSingleton);
        }

        public override void Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            IsDisposed = true;
            _magifyDispose();
            _legacyCountersStorage.Dispose();
            _onBackground.Dispose();
            _onForeground.Dispose();
            _onApplicationWillResignActive.Dispose();
            _onLocaleChanged.Dispose();
            _onNewPackageInstalled.Dispose();
            _disposable.Release();
            base.Dispose();
        }

        public override MigrationData LoadLegacyNativeDataForMigration()
        {
            var legacyDirPath = _magifyCollectMigrationData();

            if (!Directory.Exists(legacyDirPath))
            {
                return null;
            }

            var countersFilePath = Path.Combine(legacyDirPath, "counters");
            if (File.Exists(countersFilePath))
            {
                MigrateLegacyCountersToBinaryStorageFromFile(countersFilePath);
            }

            var migrationData = new MigrationData();
            FillMigrationDataFromLegacyDirectory(migrationData, legacyDirPath);

            return migrationData;
        }

        public override void CleanLegacyNativeDataAfterMigration()
        {
            _magifyClearDataAfterMigration();
        }

        [CanBeNull]
        public override int? GetGlobalCounterFromLegacyData(CounterType type, CounterKey key)
        {
            _migrationLogger.Log($"Getting global counter for type: {type}, key: {key}");
            var (scope, keyStr) = type switch
            {
                CounterType.Bonuses => (default, default),
                CounterType.Clicks => ("GlobalClickCounter", ExtractClicksCounterKeyFrom(key)),
                CounterType.Events => ("GlobalEventCounter", ExtractEventsCounterKeyFrom(key)),
                CounterType.Impressions => key.LimitId == null
                    ? ("GlobalImpressionCounter", ExtractImpressionsCounterKeyFrom(key))
                    : ("PeriodTempLimitsCounter", ExtractTempLimitsCounterKeyFrom(key)),
                CounterType.Rewards => ("GlobalRVCounter", ExtractRewardsCounterKeyFrom(key)),
                CounterType.Nested => ("GlobalNestedCampaignsCounter", ExtractNestedCounterKeyFrom(key)),
                CounterType.LtoStart => ("LtoActivationPerTriggerGlobalCounter", ExtractLtoCounterKeyFrom(key)),
                _ => (default, default),
            };

            if (scope == null) _migrationLogger.Log($"The {nameof(scope)} to get global counter is null, returning null");
            if (keyStr == null) _migrationLogger.Log($"The {nameof(keyStr)} to get global counter is null, returning null");
            if (scope == null || keyStr == null)
            {
                return null;
            }

            return _legacyCountersStorage.GetGlobalCounter(scope, keyStr);
        }

        private void MigrateLegacyCountersToBinaryStorageFromFile([NotNull] string countersFilePath)
        {
            var countersJson = File.ReadAllText(countersFilePath);
            var counters = JsonConvert.DeserializeObject<Dictionary<string, Dictionary<string, int>>>(countersJson);

            _legacyCountersStorage.MigrateCounters(counters);
        }

        private static void FillMigrationDataFromLegacyDirectory([NotNull] MigrationData migrationData, [NotNull] string legacyDirectoryPath)
        {
            var contextFilePath = Path.Combine(legacyDirectoryPath, "context");
            if (File.Exists(contextFilePath))
            {
                migrationData.ContextPath = contextFilePath;
            }

            var ltoInfoFilePath = Path.Combine(legacyDirectoryPath, "ltoInfo");
            if (File.Exists(ltoInfoFilePath))
            {
                var ltoInfoJson = File.ReadAllText(ltoInfoFilePath);
                var ltoInfos = JsonConvert.DeserializeObject<List<LtoInfoMigrationData>>(ltoInfoJson);

                migrationData.ActiveTimeLimitedOffers = ltoInfos?
                    .Where(offer => offer != null)
                    .Select(offer => new LtoModel
                    {
                        HasPriority = offer.IsPriorityLaunch,
                        Trigger = offer.LaunchedTrigger,
                        BadgePlaceholder = offer.DefaultSpotImage,
                        CampaignName = offer.CampaignName,
                        Spot = offer.Spot,
                        StartTime = DateTime.UnixEpoch + TimeSpan.FromSeconds(offer.StartTime),
                        EndTime = DateTime.UnixEpoch + TimeSpan.FromSeconds(offer.EndTime),
                    })
                    .ToList();
            }

            var cachedAnalyticsFilePath = Path.Combine(legacyDirectoryPath, "cachedAnalytics");
            if (File.Exists(cachedAnalyticsFilePath))
            {
                var cachedAnalyticsJson = File.ReadAllText(cachedAnalyticsFilePath);
                var cachedAnalytics = JsonConvert.DeserializeObject<CachedEventsMigrationData>(cachedAnalyticsJson);
                if (cachedAnalytics != null)
                {
                    migrationData.AppLaunchEvents = cachedAnalytics.AppLaunchEvents;
                    migrationData.CustomSessionEvents = cachedAnalytics.SessionEventEvents;
                    migrationData.AppBackgroundEvents = cachedAnalytics.AppBackgroundEvents;
                    migrationData.TransactionEvents = cachedAnalytics.TransactionEvents;
                    migrationData.FirebaseMappingEvents = cachedAnalytics.FirebaseMappingEvents;
                    migrationData.MailingStatusMappingEvents = cachedAnalytics.MailingStatusMappingEvents;
                    migrationData.ImpressionEvents = cachedAnalytics.ImpressionEvents;
                    migrationData.ApplovinAdsImpressionEvents = cachedAnalytics.ApplovinAdsImpressionEvents;
                    migrationData.IronSourceAdsImpressionEvents = cachedAnalytics.IronSourceAdsImpressionEvents;
                    migrationData.ImpressionFailEvents = cachedAnalytics.ImpressionFailEvents;
                    migrationData.TrialActivationEvents = cachedAnalytics.TrialActivationEvents;
                    migrationData.PaidSubscriptionActivationEvents = cachedAnalytics.PaidSubscriptionActivationEvents;
                    migrationData.InAppEvents = cachedAnalytics.InAppEvents;
                    migrationData.ClickEvents = cachedAnalytics.ClickEvents;
                    migrationData.AdsClickEvents = cachedAnalytics.AdsClickEvents;
                    migrationData.PurchasedRecords = cachedAnalytics.PurchasedRecords;
                }
            }

            var generalPrefsFilePath = Path.Combine(legacyDirectoryPath, "generalInfo");
            if (File.Exists(generalPrefsFilePath))
            {
                var generalPrefsJson = File.ReadAllText(generalPrefsFilePath);
                _migrationLogger.Log($"Native GeneralPrefs json content: \n{generalPrefsJson}");
                var generalPrefs = JsonConvert.DeserializeObject<GeneralPrefsMigrationData>(generalPrefsJson);

                if (generalPrefs == null)
                {
                    return;
                }
                var purchasedInAppProducts = generalPrefs.PurchasedInAppProducts;
                var inAppStatus = InAppStatus.NeverPurchased;
                if (purchasedInAppProducts is { Count: > 0 })
                {
                    inAppStatus = InAppStatus.Purchased;
                }

                migrationData.ClientId = generalPrefs.ClientId;
                migrationData.FirstInstalledVersion = generalPrefs.FirstInstalledVersion;
                migrationData.FirstLaunchDate = DateTime.UnixEpoch + TimeSpan.FromSeconds(generalPrefs.FirstLaunchDate);
                migrationData.InAppStatus = inAppStatus;
                migrationData.AdjustId = generalPrefs.AdjustId;
                migrationData.AttAuthorizationStatus = generalPrefs.AttAuthorizationStatus;
                migrationData.UsedDefaultCampaignTypes = generalPrefs.UsedDefaultCampaignTypes;
                migrationData.ProductsReceiptTimestamps = generalPrefs.ProductsTimestamps?.ToDictionary(
                    pair => pair.Key,
                    pair => DateTime.UnixEpoch + TimeSpan.FromSeconds(pair.Value));
                migrationData.PurchasedInAppProducts = purchasedInAppProducts;
                migrationData.PurchasedSubscriptionProducts = generalPrefs.PurchasedSubscriptionProducts;
                migrationData.UsedFreeProducts = generalPrefs.UsedFreeProducts;
                migrationData.FirebaseInstanceId = generalPrefs.FirebaseInstanceId;
                migrationData.GlobalSessionCounter = generalPrefs.GlobalSessionCounter;
                migrationData.SessionId = generalPrefs.SessionId;
                migrationData.SessionStarted = generalPrefs.SessionStarted;
                migrationData.LastActivity = generalPrefs.LastActivity;
                migrationData.MediaSource = new MediaSource(
                    generalPrefs.MediaSourceMigrationData?.NetworkName,
                    generalPrefs.MediaSourceMigrationData?.CampaignName,
                    generalPrefs.MediaSourceMigrationData?.GroupName
                );
            }
        }

        private static string ExtractClicksCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.CampaignNameProp | CounterKey.Flags.SourceProp | CounterKey.Flags.SourceTypeProp:
                    return $"{key.SourceType!.Value.ToEnumString()}-{key.CampaignName}-{key.Source}";

                case CounterKey.Flags.CampaignNameProp:
                    return key.CampaignName;
            }

            _migrationLogger.Log($"{nameof(ExtractClicksCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractEventsCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.SourceProp:
                    return key.Source;
            }

            _migrationLogger.Log($"{nameof(ExtractEventsCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractImpressionsCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.CampaignNameProp | CounterKey.Flags.SourceProp | CounterKey.Flags.SourceTypeProp:
                    switch (key.SourceType)
                    {
                        case SourceType.Event:
                            return $"{key.CampaignName}-{key.Source}";

                        case SourceType.Trigger:
                            return $"{key.CampaignName}-trigger-{key.Source}";

                        case SourceType.Placement:
                            return null;
                    }
                    break;

                case CounterKey.Flags.CampaignNameProp | CounterKey.Flags.SourceTypeProp:
                    return $"{key.CampaignName}-{key.SourceType!.Value.ToEnumString()}";

                case CounterKey.Flags.CampaignTypeProp | CounterKey.Flags.SourceTypeProp:
                    return $"{key.CampaignType!.Value.ToEnumString()}-{key.SourceType!.Value.ToEnumString()}";

                case CounterKey.Flags.CampaignTypeProp:
                    return key.CampaignType!.Value.ToEnumString();

                case CounterKey.Flags.CampaignNameProp:
                    return key.CampaignName;

                case CounterKey.Flags.SourceTypeProp:
                    return key.SourceType!.Value.ToEnumString();
            }

            _migrationLogger.Log($"{nameof(ExtractImpressionsCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractTempLimitsCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.LimitIdProp | CounterKey.Flags.CampaignTypeProp:
                    return $"tempLimits-{key.LimitId}-{key.CampaignType!.Value.ToEnumString()}";
            }

            _migrationLogger.Log($"{nameof(ExtractTempLimitsCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractRewardsCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.CampaignNameProp | CounterKey.Flags.SourceProp | CounterKey.Flags.SourceTypeProp:
                    return $"rv-{key.SourceType!.Value.ToEnumString()}-{key.CampaignName}-{key.Source}";

                case CounterKey.Flags.CampaignNameProp:
                    return key.CampaignName;
            }

            _migrationLogger.Log($"{nameof(ExtractRewardsCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractNestedCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.NestedNameProp:
                    return key.NestedName;
            }

            _migrationLogger.Log($"{nameof(ExtractNestedCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }

        private static string ExtractLtoCounterKeyFrom(CounterKey key)
        {
            switch (key.GetUsedProperties())
            {
                case CounterKey.Flags.CampaignNameProp | CounterKey.Flags.SourceProp:
                    return $"{key.CampaignName}-{key.Source}";

                case CounterKey.Flags.CampaignNameProp:
                    return $"lto_actiovation_counter_{key.CampaignName}";
            }

            _migrationLogger.Log($"{nameof(ExtractLtoCounterKeyFrom)} called for key: {key}, flags: {key.GetUsedProperties()}");
            return null;
        }
    }
}
#endif