using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        [NotNull, ItemNotNull]
        private static readonly string[] _restoreProgressExpectedHeaders = new string[]
        {
            "date",
        };

        public UniTask<RestoreStateRequestResponse> RestoreState([CanBeNull] int? weight, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                ApplicationStateApiMethods.GetProgress(_platform.RuntimePlatform),
                createRestoreStateRequestPayload,
                useWebRequestWithRepeats: false,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestRestoreStateContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: RestoreStateRequestResponse.Fail());

            WebRequestMessage createRestoreStateRequestPayload()
            {
                ThrowIfDisposed();
                return new WebRequestMessage
                {
                    AuthToken = default,
                    Method = ApplicationStateApiMethods.GetProgress(_platform.RuntimePlatform),
                    Payload = new RestoreStatePayload
                    {
                        Weight = weight,
                        LastSyncTime = ShouldSendLastSyncProgressTime ? _appStatePrefs.LastSyncTime.Value : null,
                    },
                    ExpectedHeaders = _restoreProgressExpectedHeaders,
                };
            }
        }

        private RepeatState RestoreStateResponseHandler(WebResponseMessage response, out RestoreStateRequestResponse result)
        {
            _logger.Log($"{nameof(RestoreStateResponseHandler)} called");
            if (response.Result == UnityWebRequest.Result.Success && !string.IsNullOrEmpty(response.Text))
            {
                _logger.Log($"State successfully restored. Result text={response.Text}");
                if (ServerApiUtils.TryDeserializeResponse<GaladrielResponse<AppStateStateResponse>>(response.Text, _logger, out var state, out var exception))
                {
                    result = RestoreStateRequestResponse.Success(state.Result?.Progress, state.Result?.Weight);
                    if (response.TryGetUnixTimeSeconds(out var unixTimeMs))
                    {
                        _appStatePrefs.LastSyncTime.Value = unixTimeMs;
                    }
                    return RepeatState.Finish;
                }
                else
                {
                    throw new MagifyFailedToParseRestoreProgressResponseException(response.Text, exception);
                }
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    _logger.Log($"Default error code detected for: {errorResponse?.Error}");
                    result = RestoreStateRequestResponse.Fail();
                    return repeatState;
                }
                switch (errorResponse?.Error?.Code)
                {

                    case ErrorCode.SyncProgress_AlreadyHasNewest:
                        _logger.Log("An already the newest progress error was received while requesting to get progress");
                        result = RestoreStateRequestResponse.AlreadyHasNewest();
                        return RepeatState.Finish;

                    case ErrorCode.SyncProgress_OnlyForceRestorePossible:
                        _logger.Log("An only force get progress possible error was received while requesting to get progress");
                        result = RestoreStateRequestResponse.OnlyForceRestorePossible();
                        return RepeatState.Finish;
                }
            }

            result = RestoreStateRequestResponse.Fail();
            return RepeatState.Finish;
        }

        public void CancelAllRestoreStateRequests()
        {
            _logger.Log($"{nameof(CancelAllRestoreStateRequests)} has been called");
            _requestRestoreStateContext.CancelAllRequests();
        }
    }
}